# 🔧 Fix "Invalid Credentials" Login Issue

## Problem
You're getting `{"error":"Invalid credentials"}` when trying to login because the SQL file contains placeholder password hashes that don't match the actual passwords.

---

## ⚡ Quick Fix (2 Steps)

### Step 1: Upload the Script to Server
Upload this file to your server:
```
backend/scripts/hashPasswords.js
```

### Step 2: Run the Script on Server
```bash
cd /root/public_html/backend
node scripts/hashPasswords.js
```

This will:
- ✅ Properly hash the admin passwords with bcrypt
- ✅ Update the database with correct password hashes
- ✅ Test the passwords to verify they work
- ✅ Show you the login credentials

---

## 🔍 What Went Wrong?

The SQL file has this placeholder:
```sql
INSERT INTO users (email, password, ...) VALUES
('admin@faceml.com', '$2a$10$YourHashedPasswordHere', ...);
```

The `$2a$10$YourHashedPasswordHere` is just a placeholder text, not a real bcrypt hash of "Admin@123".

**You MUST run the script to generate proper bcrypt hashes!**

---

## 📋 Expected Output After Running Script

```
🔄 Connecting to MySQL database...
✓ Connected to database

🔄 Hashing Super Admin password...
✓ Super Admin password updated

🔄 Hashing Shop Admin password...
✓ Shop Admin password updated

✅ Password hashing completed successfully!

═══════════════════════════════════════
📋 Login Credentials:
═══════════════════════════════════════

🔑 Super Admin:
   Email: admin@faceml.com
   Password: Admin@123

🔑 Shop Admin:
   Email: shop@gmail.com
   Password: 123456

═══════════════════════════════════════
⚠️  IMPORTANT: Change these passwords after first login!
═══════════════════════════════════════

🧪 Testing Super Admin login...
✓ Super Admin password verification successful!

🧪 Testing Shop Admin login...
✓ Shop Admin password verification successful!

✅ All tests passed! You can now login via API.
```

---

## ✅ Test Login After Fix

### Test 1: Super Admin
```bash
curl --location 'http://180.92.197.145:3200/api/auth/login' \
--header 'Content-Type: application/json' \
--data-raw '{
    "email": "admin@faceml.com",
    "password": "Admin@123"
}'
```

**Expected:** JSON with `token` field ✓

### Test 2: Shop Admin
```bash
curl --location 'http://180.92.197.145:3200/api/auth/login' \
--header 'Content-Type: application/json' \
--data-raw '{
    "email": "shop@gmail.com",
    "password": "123456"
}'
```

**Expected:** JSON with `token` field ✓

---

## 🔐 Check Your .env File

Make sure your `.env` file on the server has these values:

```env
# Database Configuration
DB_HOST=localhost
DB_USER=your_mysql_user
DB_PASSWORD=your_mysql_password
DB_NAME=your_database_name
DB_PORT=3306

# Server Configuration
PORT=3200

# JWT Secret (REQUIRED - Generate a random string)
JWT_SECRET=your_long_random_secret_key_at_least_32_characters

# Admin Credentials (used by hashPasswords.js)
SUPER_ADMIN_EMAIL=admin@faceml.com
SUPER_ADMIN_PASSWORD=Admin@123

# Shop Admin (optional, for demo)
SHOP_ADMIN_EMAIL=shop@gmail.com
SHOP_ADMIN_PASSWORD=123456
```

**IMPORTANT:** If you don't have `JWT_SECRET`, generate one:
```bash
node -e "console.log(require('crypto').randomBytes(32).toString('hex'))"
```

---

## 🐛 Troubleshooting

### Error: "Cannot find module"
**Fix:**
```bash
cd /root/public_html/backend
npm install bcryptjs mysql2 dotenv
node scripts/hashPasswords.js
```

### Error: "Database connection failed"
**Fix:**
1. Check `.env` file has correct DB credentials
2. Test MySQL connection:
```bash
mysql -u your_user -p your_database
```
3. Verify database was imported:
```bash
mysql -u your_user -p your_database -e "SHOW TABLES;"
```

### Error: "Access denied for user"
**Fix:**
Update `.env` with correct MySQL username and password from cPanel

### Still getting "Invalid credentials" after running script
**Fix:**
1. Check JWT_SECRET is set in `.env`
2. Restart your Node.js server:
```bash
pm2 restart faceml
# or
pm2 restart all
```
3. Check server logs:
```bash
pm2 logs faceml
```

---

## 📝 Alternative: Manual Password Hash

If you can't run the script, you can manually hash passwords:

### Generate Hash
```bash
node -e "const bcrypt = require('bcryptjs'); bcrypt.hash('Admin@123', 10).then(hash => console.log(hash));"
```

### Update Database Manually
```bash
mysql -u your_user -p your_database
```

```sql
UPDATE users 
SET password = 'paste_hash_here' 
WHERE email = 'admin@faceml.com';

-- Check it worked
SELECT email, LEFT(password, 20) as password_preview FROM users;
```

---

## 🎯 Summary

**The issue:** SQL file has placeholder password hashes  
**The fix:** Run `node scripts/hashPasswords.js` to generate real hashes  
**Time needed:** 1 minute  

**DO THIS NOW:**
```bash
cd /root/public_html/backend
node scripts/hashPasswords.js
```

Then test login again! 🚀

---

## ✅ Success Checklist

After running the fix:
- [ ] Script shows "✓ Super Admin password updated"
- [ ] Script shows "✓ Shop Admin password updated"
- [ ] Script shows "✅ All tests passed!"
- [ ] Can login via curl/Postman
- [ ] Can login via Flutter app
- [ ] Server is running (pm2 status shows 'online')

---

**Need help?** Check the script output for specific error messages.

