# 🔐 Fix Admin Password After Database Import

If you're getting "Invalid credentials" error after importing the database, it means the password hash needs to be updated.

## 🚨 Problem

After importing the SQL dump, the admin user has a placeholder password hash (`$2a$10$YourHashedPasswordHere`) which cannot be verified. You need to generate a proper bcrypt hash.

## ✅ Solution

### Option 1: Use the Fix Password Script (Recommended)

1. **SSH into your server** (or access your server terminal)

2. **Navigate to backend directory**:
   ```bash
   cd /path/to/your/backend
   ```

3. **Ensure .env file is configured** with your database credentials:
   ```env
   DB_HOST=localhost
   DB_USER=your_cpanel_db_user
   DB_PASSWORD=your_db_password
   DB_NAME=your_cpanel_db_name
   DB_PORT=3306
   
   SUPER_ADMIN_EMAIL=admin@faceml.com
   SUPER_ADMIN_PASSWORD=Admin@123
   ```

4. **Run the fix script**:
   ```bash
   node scripts/fixAdminPassword.js
   ```

5. **Test login**:
   ```bash
   curl --location 'http://your-server:3200/api/auth/login' \
   --header 'Content-Type: application/json' \
   --data-raw '{
       "email": "admin@faceml.com",
       "password": "Admin@123"
   }'
   ```

### Option 2: Manual SQL Update (If script doesn't work)

1. **Generate password hash** using Node.js:
   ```bash
   node -e "const bcrypt = require('bcryptjs'); bcrypt.hash('Admin@123', 10).then(hash => console.log(hash));"
   ```

2. **Copy the hash** (it will look like: `$2a$10$...`)

3. **Update in phpMyAdmin or via MySQL command**:
   ```sql
   UPDATE users 
   SET password = 'YOUR_GENERATED_HASH_HERE' 
   WHERE email = 'admin@faceml.com';
   ```

### Option 3: Use initDatabase.js (Alternative)

The `initDatabase.js` script can also fix the password, but it uses `ON DUPLICATE KEY UPDATE` which won't update existing passwords. However, you can modify it temporarily:

1. **Edit** `scripts/initDatabase.js`
2. **Change line 146** from:
   ```javascript
   ON DUPLICATE KEY UPDATE updated_at = CURRENT_TIMESTAMP;
   ```
   to:
   ```javascript
   ON DUPLICATE KEY UPDATE password = VALUES(password), updated_at = CURRENT_TIMESTAMP;
   ```
3. **Run** `node scripts/initDatabase.js`

## 🔍 Troubleshooting

### "Cannot find module 'bcryptjs'"
```bash
npm install bcryptjs
```

### "Cannot find module 'mysql2'"
```bash
npm install mysql2
```

### "Access denied" error
- Check your `.env` file database credentials
- Verify database user has proper permissions
- Make sure you're using the **full** cPanel database username (with prefix)

### "Database does not exist"
- Verify the database name in `.env` matches your cPanel database name
- Remember cPanel adds prefixes: `cpses_xxxxx_faceml`

### Still getting "Invalid credentials"
1. Verify the password hash was updated:
   ```sql
   SELECT email, password FROM users WHERE email = 'admin@faceml.com';
   ```
   The password should be a long hash starting with `$2a$10$`, NOT `$2a$10$YourHashedPasswordHere`

2. Double-check you're using the correct password in your login request

3. Verify the user exists and is active:
   ```sql
   SELECT email, is_active FROM users WHERE email = 'admin@faceml.com';
   ```

## 📝 Quick Command Reference

```bash
# Fix password
node scripts/fixAdminPassword.js

# Test login
curl -X POST http://your-server:3200/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@faceml.com","password":"Admin@123"}'

# Check user in database (via MySQL CLI)
mysql -u your_user -p your_database
SELECT email, LEFT(password, 30) as password_preview FROM users WHERE email = 'admin@faceml.com';
```

## ✅ Success Indicators

After running the fix script, you should see:
```
✓ Connected to database
✓ Found user: admin@faceml.com
🔄 Updating password hash...
✓ Password hash updated successfully!
✅ Password fix completed!

Login Credentials:
  Email: admin@faceml.com
  Password: Admin@123
```

Then test login should return a token instead of "Invalid credentials".

## 🔒 Security Note

**IMPORTANT**: After successfully logging in, immediately change the default password through the app's change password feature!

