const db = require('../config/database');
const encryptionService = require('../services/encryptionService');

// Clock in
exports.clockIn = async (req, res) => {
  try {
    let { staff_id: staffId, branch_id: branchId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === 'staff') {
      staffId = req.user.staff_id;
      branchId = req.user.branch_id;
    }

    console.log('\n========================================');
    console.log('⏰ CLOCK IN REQUEST');
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`🏢 Branch ID: ${branchId}`);
    console.log(`📍 Location: ${location || 'Not provided'}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log('========================================\n');

    if (!staffId || !branchId) {
      return res.status(400).json({ error: 'Staff ID and branch ID are required' });
    }

    // Verify staff belongs to this admin's branch
    let staff;
    if (requesterRole === 'staff') {
      [staff] = await db.query(
        `
        SELECT st.id, st.full_name
        FROM staff st
        WHERE st.id = ? AND st.branch_id = ?
      `,
        [staffId, branchId],
      );
    } else {
      [staff] = await db.query(`
        SELECT st.id, st.full_name
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND b.id = ? AND s.admin_id = ?
      `, [staffId, branchId, req.user.id]);
    }

    if (staff.length === 0) {
      console.log('❌ Access denied - staff not found or doesn\'t belong to admin\'s branch\n');
      return res.status(403).json({ error: 'Access denied or staff not found' });
    }

    console.log(`✅ Staff verified: ${staff[0].full_name}`);

    const today = new Date().toISOString().split('T')[0];
    console.log(`📆 Today's date: ${today}`);

    // Check if already clocked in today
    // NOTE: Each day has its own attendance_date, so night shifts that span two days work correctly:
    // - Day 1 (8pm): attendance_date = "2024-01-15"
    // - Day 2 (8am auto-clockout): clock_out_time set, but attendance_date stays "2024-01-15"
    // - Day 2 (8pm new shift): attendance_date = "2024-01-16" (different date, so clock-in allowed)
    const [existing] = await db.query(
      `SELECT id, clock_out_time FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?`,
      [staffId, today]
    );

    console.log(`📋 Existing attendance records for today: ${existing.length}`);
    if (existing.length > 0) {
      console.log(`   - Record ID: ${existing[0].id}`);
      console.log(`   - Clock Out Time: ${existing[0].clock_out_time || 'NULL (still clocked in)'}`);
    }

    // Only reject if there's a record for TODAY with no clock_out_time
    // If clock_out_time exists (even from auto-clockout), allow new clock-in for the same day
    if (existing.length > 0 && !existing[0].clock_out_time) {
      console.log('❌ Already clocked in for today\n');
      return res.status(400).json({ error: 'Already clocked in for today' });
    }

    // Get local time string from client or use current time as string
    // Format: YYYY-MM-DD HH:MM:SS (local time, no timezone conversion)
    let clockInTimeStr = req.body.clock_in_time;
    if (!clockInTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, '0');
      const day = String(now.getDate()).padStart(2, '0');
      const hours = String(now.getHours()).padStart(2, '0');
      const minutes = String(now.getMinutes()).padStart(2, '0');
      const seconds = String(now.getSeconds()).padStart(2, '0');
      clockInTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Create new attendance record - store time as string
    const [result] = await db.query(
      `INSERT INTO attendance 
       (staff_id, branch_id, clock_in_time, clock_in_location, attendance_date, status) 
       VALUES (?, ?, ?, ?, ?, 'present')`,
      [staffId, branchId, clockInTimeStr, location || null, today]
    );

    console.log(`✅ Clock in successful!`);
    console.log(`   - Attendance ID: ${result.insertId}`);
    console.log(`   - Staff: ${staff[0].full_name}`);
    console.log(`   - Date: ${today}`);
    console.log(`   - Clock In Time (stored as string): ${clockInTimeStr}`);
    console.log('========================================\n');

    res.status(201).json({
      message: 'Clocked in successfully',
      attendance: {
        id: result.insertId,
        staff_id: staffId,
        branch_id: branchId,
        clock_in_time: clockInTimeStr,
        attendance_date: today
      }
    });

  } catch (error) {
    console.error('❌ Clock in error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Clock out
exports.clockOut = async (req, res) => {
  try {
    let { staff_id: staffId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === 'staff') {
      staffId = req.user.staff_id;
    }

    if (!staffId) {
      return res.status(400).json({ error: 'Staff ID is required' });
    }

    // Verify staff belongs to this admin
    let staff;
    if (requesterRole === 'staff') {
      [staff] = await db.query(
        `
        SELECT st.id, st.branch_id
        FROM staff st
        WHERE st.id = ?
      `,
        [staffId],
      );
    } else {
      [staff] = await db.query(`
        SELECT st.id
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND s.admin_id = ?
      `, [staffId, req.user.id]);
    }

    if (staff.length === 0) {
      return res.status(403).json({ error: 'Access denied or staff not found' });
    }

    const today = new Date().toISOString().split('T')[0];
    console.log('\n========================================');
    console.log('⏰ CLOCK OUT REQUEST');
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`📍 Location: ${location || 'Not provided'}`);
    console.log(`📆 Today's date: ${today}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log('========================================\n');

    // Find today's attendance record
    const [attendance] = await db.query(
      `SELECT id, clock_in_time, clock_out_time, attendance_date
       FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?
       ORDER BY clock_in_time DESC
       LIMIT 1`,
      [staffId, today]
    );

    console.log(`📋 Attendance records found: ${attendance.length}`);
    
    if (attendance.length === 0) {
      console.log('❌ No clock-in record found for today\n');
      return res.status(400).json({ error: 'No clock-in record found for today' });
    }

    const record = attendance[0];
    
    // Check if already clocked out - handle NULL properly
    // In MySQL, NULL values need to be checked explicitly
    const existingClockOutTime = record.clock_out_time;
    
    console.log(`📝 Record details:`);
    console.log(`   - Record ID: ${record.id}`);
    console.log(`   - Clock In: ${record.clock_in_time}`);
    console.log(`   - Clock Out: ${existingClockOutTime || 'NULL (not clocked out)'}`);
    console.log(`   - Attendance Date: ${record.attendance_date}`);
    console.log(`   - Clock Out Type: ${typeof existingClockOutTime}`);
    console.log(`   - Clock Out Value: ${JSON.stringify(existingClockOutTime)}`);
    const isClockOutNull = existingClockOutTime === null || 
                           existingClockOutTime === undefined || 
                           existingClockOutTime === 'null' || 
                           existingClockOutTime === '' ||
                           (typeof existingClockOutTime === 'string' && 
                            (existingClockOutTime.trim() === '' || 
                             existingClockOutTime.toLowerCase() === 'null' ||
                             existingClockOutTime.startsWith('0000-00-00')));
    
    if (!isClockOutNull) {
      console.log('❌ Already clocked out for today\n');
      return res.status(400).json({ error: 'Already clocked out for today' });
    }

    console.log('✅ Clock out allowed - record found with no clock_out_time');

    // Get local time string from client or use current time as string
    let clockOutTimeStr = req.body.clock_out_time;
    if (!clockOutTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, '0');
      const day = String(now.getDate()).padStart(2, '0');
      const hours = String(now.getHours()).padStart(2, '0');
      const minutes = String(now.getMinutes()).padStart(2, '0');
      const seconds = String(now.getSeconds()).padStart(2, '0');
      clockOutTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Calculate total hours (subtract break time)
    // Parse time strings to DateTime for calculation
    const clockInTime = new Date(record.clock_in_time);
    const clockOutTime = new Date(clockOutTimeStr);
    const totalTimeMs = clockOutTime - clockInTime;
    
    // Get total break time for today
    const [breaks] = await db.query(
      `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
       FROM work_sessions
       WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
      [record.id]
    );
    
    const totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
    const totalBreakMs = totalBreakMinutes * 60 * 1000;
    const totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);

    console.log(`⏱️  Calculating hours:`);
    console.log(`   - Clock In: ${record.clock_in_time}`);
    console.log(`   - Clock Out: ${clockOutTimeStr}`);
    console.log(`   - Total Time: ${(totalTimeMs / (1000 * 60 * 60)).toFixed(2)} hours`);
    console.log(`   - Break Time: ${totalBreakMinutes} minutes`);
    console.log(`   - Net Hours: ${totalHours.toFixed(2)} hours`);

    // Update attendance record - store time as string
    const [updateResult] = await db.query(
      `UPDATE attendance 
       SET clock_out_time = ?, 
           clock_out_location = ?, 
           total_hours = ?
       WHERE id = ?`,
      [clockOutTimeStr, location || null, totalHours.toFixed(2), record.id]
    );

    console.log(`✅ Clock out successful!`);
    console.log(`   - Attendance ID: ${record.id}`);
    console.log(`   - Clock Out Time (stored as string): ${clockOutTimeStr}`);
    console.log(`   - Total Hours: ${totalHours.toFixed(2)}`);
    console.log('========================================\n');

    res.json({
      message: 'Clocked out successfully',
      attendance: {
        id: record.id,
        clock_out_time: clockOutTimeStr,
        total_hours: totalHours.toFixed(2)
      }
    });

  } catch (error) {
    console.error('Clock out error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Break clock in
exports.breakClockIn = async (req, res) => {
  try {
    let { staff_id: staffId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === 'staff') {
      staffId = req.user.staff_id;
    }

    console.log('\n========================================');
    console.log('☕ BREAK CLOCK IN REQUEST');
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`📍 Location: ${location || 'Not provided'}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log('========================================\n');

    if (!staffId) {
      return res.status(400).json({ error: 'Staff ID is required' });
    }

    const today = new Date().toISOString().split('T')[0];

    // Find today's attendance record
    const [attendance] = await db.query(
      `SELECT id, clock_in_time, clock_out_time
       FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?
       ORDER BY clock_in_time DESC
       LIMIT 1`,
      [staffId, today]
    );

    if (attendance.length === 0) {
      console.log('❌ No clock-in record found for today\n');
      return res.status(400).json({ error: 'Please clock in first before taking a break' });
    }

    const record = attendance[0];

    // Check if already clocked out
    if (record.clock_out_time) {
      console.log('❌ Already clocked out for today\n');
      return res.status(400).json({ error: 'Cannot take break after clocking out' });
    }

    // Check if there's an active break (break clocked in but not clocked out)
    const [activeBreaks] = await db.query(
      `SELECT id FROM work_sessions
       WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NULL
       ORDER BY start_time DESC
       LIMIT 1`,
      [record.id]
    );

    if (activeBreaks.length > 0) {
      console.log('❌ Already on break\n');
      return res.status(400).json({ error: 'Already on break. Please clock out from break first' });
    }

    // Get local time string from client or use current time as string
    let breakStartTimeStr = req.body.break_start_time;
    if (!breakStartTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, '0');
      const day = String(now.getDate()).padStart(2, '0');
      const hours = String(now.getHours()).padStart(2, '0');
      const minutes = String(now.getMinutes()).padStart(2, '0');
      const seconds = String(now.getSeconds()).padStart(2, '0');
      breakStartTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Create break session - store time as string
    const [result] = await db.query(
      `INSERT INTO work_sessions 
       (attendance_id, staff_id, start_time, session_type) 
       VALUES (?, ?, ?, 'break')`,
      [record.id, staffId, breakStartTimeStr]
    );

    console.log(`✅ Break clock in successful!`);
    console.log(`   - Break Session ID: ${result.insertId}`);
    console.log(`   - Attendance ID: ${record.id}`);
    console.log(`   - Break Start Time (stored as string): ${breakStartTimeStr}`);
    console.log('========================================\n');

    res.status(201).json({
      message: 'Break started successfully',
      break_session: {
        id: result.insertId,
        attendance_id: record.id,
        start_time: breakStartTimeStr
      }
    });

  } catch (error) {
    console.error('❌ Break clock in error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Break clock out
exports.breakClockOut = async (req, res) => {
  try {
    let { staff_id: staffId, location } = req.body;
    const requesterRole = req.user.role;

    if (requesterRole === 'staff') {
      staffId = req.user.staff_id;
    }

    console.log('\n========================================');
    console.log('☕ BREAK CLOCK OUT REQUEST');
    console.log(`👤 Staff ID: ${staffId}`);
    console.log(`📍 Location: ${location || 'Not provided'}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log('========================================\n');

    if (!staffId) {
      return res.status(400).json({ error: 'Staff ID is required' });
    }

    const today = new Date().toISOString().split('T')[0];

    // Find today's attendance record
    const [attendance] = await db.query(
      `SELECT id FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?
       ORDER BY clock_in_time DESC
       LIMIT 1`,
      [staffId, today]
    );

    if (attendance.length === 0) {
      console.log('❌ No attendance record found for today\n');
      return res.status(400).json({ error: 'No attendance record found for today' });
    }

    const record = attendance[0];

    // Find active break session
    const [activeBreaks] = await db.query(
      `SELECT id, start_time FROM work_sessions
       WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NULL
       ORDER BY start_time DESC
       LIMIT 1`,
      [record.id]
    );

    if (activeBreaks.length === 0) {
      console.log('❌ No active break found\n');
      return res.status(400).json({ error: 'No active break found. Please clock in for break first' });
    }

    const breakSession = activeBreaks[0];
    
    // Get local time string from client or use current time as string
    let breakEndTimeStr = req.body.break_end_time;
    if (!breakEndTimeStr) {
      // If client didn't send time, use current server time as string
      const now = new Date();
      const year = now.getFullYear();
      const month = String(now.getMonth() + 1).padStart(2, '0');
      const day = String(now.getDate()).padStart(2, '0');
      const hours = String(now.getHours()).padStart(2, '0');
      const minutes = String(now.getMinutes()).padStart(2, '0');
      const seconds = String(now.getSeconds()).padStart(2, '0');
      breakEndTimeStr = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    }

    // Calculate duration for display (parse strings to Date for calculation)
    const breakStartTime = new Date(String(breakSession.start_time).replace(' ', 'T'));
    const breakEndTime = new Date(breakEndTimeStr.replace(' ', 'T'));
    const breakDurationMinutes = Math.round((breakEndTime - breakStartTime) / (1000 * 60));

    // Update break session - store time as string
    await db.query(
      `UPDATE work_sessions 
       SET end_time = ?, duration_minutes = ?
       WHERE id = ?`,
      [breakEndTimeStr, breakDurationMinutes, breakSession.id]
    );

    console.log(`✅ Break clock out successful!`);
    console.log(`   - Break Session ID: ${breakSession.id}`);
    console.log(`   - Break End Time (stored as string): ${breakEndTimeStr}`);
    console.log(`   - Break Duration: ${breakDurationMinutes} minutes`);
    console.log('========================================\n');

    res.json({
      message: 'Break ended successfully',
      break_session: {
        id: breakSession.id,
        end_time: breakEndTime.toISOString(),
        duration_minutes: breakDurationMinutes
      }
    });

  } catch (error) {
    console.error('❌ Break clock out error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get attendance by branch and date range
exports.getAttendance = async (req, res) => {
  try {
    const { branch_id } = req.params;
    const { start_date, end_date } = req.query;

    // Verify branch belongs to this admin
    const [branches] = await db.query(`
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `, [branch_id, req.user.id]);

    if (branches.length === 0) {
      return res.status(403).json({ error: 'Access denied to this branch' });
    }

    let query = `
      SELECT 
        a.*,
        s.staff_code,
        s.full_name,
        s.position
      FROM attendance a
      JOIN staff s ON a.staff_id = s.id
      WHERE a.branch_id = ?
    `;
    
    const params = [branch_id];

    if (start_date) {
      query += ' AND a.attendance_date >= ?';
      params.push(start_date);
    }

    if (end_date) {
      query += ' AND a.attendance_date <= ?';
      params.push(end_date);
    }

    query += ' ORDER BY a.attendance_date DESC, a.clock_in_time DESC';

    const [attendance] = await db.query(query, params);

    // Decrypt staff names
    const attendanceWithDecryptedNames = attendance.map(record => {
      try {
        if (record.full_name) {
          record.full_name = encryptionService.decrypt(record.full_name);
        }
      } catch (error) {
        console.warn(`Failed to decrypt name for staff in attendance record ${record.id}:`, error.message);
      }
      return record;
    });

    res.json({ attendance: attendanceWithDecryptedNames });

  } catch (error) {
    console.error('Get attendance error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get staff attendance by staff ID
exports.getStaffAttendance = async (req, res) => {
  try {
    const { staff_id } = req.params;
    const { start_date, end_date } = req.query;

    // Verify staff belongs to this admin
    const [staff] = await db.query(`
      SELECT st.id, st.full_name, st.staff_code, b.name as branch_name
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND s.admin_id = ?
    `, [staff_id, req.user.id]);

    if (staff.length === 0) {
      return res.status(403).json({ error: 'Access denied or staff not found' });
    }

    let query = `
      SELECT *
      FROM attendance
      WHERE staff_id = ?
    `;
    
    const params = [staff_id];

    if (start_date) {
      query += ' AND attendance_date >= ?';
      params.push(start_date);
    }

    if (end_date) {
      query += ' AND attendance_date <= ?';
      params.push(end_date);
    }

    query += ' ORDER BY attendance_date DESC';

    const [attendance] = await db.query(query, params);

    // Decrypt staff name
    if (staff[0].full_name) {
      try {
        staff[0].full_name = encryptionService.decrypt(staff[0].full_name);
      } catch (error) {
        console.warn(`Failed to decrypt name for staff ${staff_id}:`, error.message);
      }
    }

    res.json({ 
      staff: staff[0],
      attendance 
    });

  } catch (error) {
    console.error('Get staff attendance error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get attendance summary/statistics
exports.getAttendanceSummary = async (req, res) => {
  try {
    const { branch_id } = req.params;
    const { start_date, end_date } = req.query;

    // Verify branch belongs to this admin
    const [branches] = await db.query(`
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `, [branch_id, req.user.id]);

    if (branches.length === 0) {
      return res.status(403).json({ error: 'Access denied to this branch' });
    }

    const today = new Date().toISOString().split('T')[0];
    const dateCondition = start_date && end_date 
      ? 'AND a.attendance_date BETWEEN ? AND ?' 
      : 'AND a.attendance_date = ?';
    
    const dateParams = start_date && end_date 
      ? [branch_id, start_date, end_date] 
      : [branch_id, today];

    // Get summary statistics
    const [summary] = await db.query(`
      SELECT 
        COUNT(DISTINCT s.id) as total_staff,
        COUNT(DISTINCT CASE WHEN a.status = 'present' THEN a.staff_id END) as present_count,
        COUNT(DISTINCT CASE WHEN a.status = 'absent' THEN a.staff_id END) as absent_count,
        AVG(a.total_hours) as avg_hours,
        SUM(a.total_hours) as total_hours
      FROM staff s
      LEFT JOIN attendance a ON s.id = a.staff_id ${dateCondition}
      WHERE s.branch_id = ? AND s.is_active = true
    `, dateParams);

    res.json({ summary: summary[0] });

  } catch (error) {
    console.error('Get attendance summary error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Mark manual attendance (for absent, leave, etc.)
exports.markManualAttendance = async (req, res) => {
  try {
    const { staff_id, branch_id, attendance_date, status, notes } = req.body;

    if (!staff_id || !branch_id || !attendance_date || !status) {
      return res.status(400).json({ 
        error: 'Staff ID, branch ID, date, and status are required' 
      });
    }

    // Verify staff belongs to this admin
    const [staff] = await db.query(`
      SELECT st.id
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND b.id = ? AND s.admin_id = ?
    `, [staff_id, branch_id, req.user.id]);

    if (staff.length === 0) {
      return res.status(403).json({ error: 'Access denied or staff not found' });
    }

    // Check if attendance already exists
    const [existing] = await db.query(
      'SELECT id FROM attendance WHERE staff_id = ? AND attendance_date = ?',
      [staff_id, attendance_date]
    );

    if (existing.length > 0) {
      // Update existing
      await db.query(
        'UPDATE attendance SET status = ?, notes = ? WHERE id = ?',
        [status, notes || null, existing[0].id]
      );

      res.json({ message: 'Attendance updated successfully' });
    } else {
      // Create new
      await db.query(
        `INSERT INTO attendance 
         (staff_id, branch_id, attendance_date, status, notes) 
         VALUES (?, ?, ?, ?, ?)`,
        [staff_id, branch_id, attendance_date, status, notes || null]
      );

      res.status(201).json({ message: 'Attendance marked successfully' });
    }

  } catch (error) {
    console.error('Mark manual attendance error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Create or update manual attendance with clock-in/clock-out times
exports.createOrUpdateManualAttendance = async (req, res) => {
  try {
    const { staff_id, branch_id, attendance_date, clock_in_time, clock_out_time, location } = req.body;

    console.log('\n========================================');
    console.log('📝 MANUAL ATTENDANCE REQUEST');
    console.log(`👤 Staff ID: ${staff_id}`);
    console.log(`🏢 Branch ID: ${branch_id}`);
    console.log(`📅 Date: ${attendance_date}`);
    console.log(`⏰ Clock In: ${clock_in_time || 'Not provided'}`);
    console.log(`⏰ Clock Out: ${clock_out_time || 'Not provided'}`);
    console.log('========================================\n');

    if (!staff_id || !branch_id || !attendance_date) {
      return res.status(400).json({ 
        error: 'Staff ID, branch ID, and attendance date are required' 
      });
    }

    if (!clock_in_time && !clock_out_time) {
      return res.status(400).json({ 
        error: 'At least clock-in or clock-out time must be provided' 
      });
    }

    // Verify staff belongs to this admin
    const [staff] = await db.query(`
      SELECT st.id, st.full_name
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND b.id = ? AND s.admin_id = ?
    `, [staff_id, branch_id, req.user.id]);

    if (staff.length === 0) {
      return res.status(403).json({ error: 'Access denied or staff not found' });
    }

    // Check if attendance already exists for this date
    const [existing] = await db.query(
      `SELECT id, clock_in_time, clock_out_time, total_hours 
       FROM attendance 
       WHERE staff_id = ? AND attendance_date = ?`,
      [staff_id, attendance_date]
    );

    // Accept time strings directly (format: YYYY-MM-DD HH:MM:SS)
    // No timezone conversion - store as-is
    let clockInTimeStr = clock_in_time ? String(clock_in_time).trim() : null;
    let clockOutTimeStr = clock_out_time ? String(clock_out_time).trim() : null;
    let totalHours = null;

    // Validate time string format (YYYY-MM-DD HH:MM:SS or YYYY-MM-DDTHH:MM:SS)
    if (clockInTimeStr) {
      // Normalize format: convert T to space if needed
      clockInTimeStr = clockInTimeStr.replace('T', ' ');
      // Validate basic format
      if (!/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(clockInTimeStr)) {
        return res.status(400).json({ error: 'Invalid clock-in time format. Expected: YYYY-MM-DD HH:MM:SS' });
      }
    }

    if (clockOutTimeStr) {
      // Normalize format: convert T to space if needed
      clockOutTimeStr = clockOutTimeStr.replace('T', ' ');
      // Validate basic format
      if (!/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(clockOutTimeStr)) {
        return res.status(400).json({ error: 'Invalid clock-out time format. Expected: YYYY-MM-DD HH:MM:SS' });
      }
    }

    // Calculate total hours if both times are provided (subtract break time)
    // Parse time strings to Date objects for calculation only
    let clockInDateTime = clockInTimeStr ? new Date(clockInTimeStr.replace(' ', 'T')) : null;
    let clockOutDateTime = clockOutTimeStr ? new Date(clockOutTimeStr.replace(' ', 'T')) : null;
    let attendanceIdForBreaks = existing.length > 0 ? existing[0].id : null;
    
    if (clockInDateTime && clockOutDateTime) {
      if (isNaN(clockInDateTime.getTime()) || isNaN(clockOutDateTime.getTime())) {
        return res.status(400).json({ error: 'Invalid time format for calculation' });
      }
      if (clockOutDateTime < clockInDateTime) {
        return res.status(400).json({ error: 'Clock-out time must be after clock-in time' });
      }
      const totalTimeMs = clockOutDateTime - clockInDateTime;
      
      // Get total break time if attendance record exists
      let totalBreakMinutes = 0;
      if (attendanceIdForBreaks) {
        const [breaks] = await db.query(
          `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
           FROM work_sessions
           WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
          [attendanceIdForBreaks]
        );
        totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
      }
      
      const totalBreakMs = totalBreakMinutes * 60 * 1000;
      totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);
    } else if (existing.length > 0 && existing[0].clock_in_time && clockOutDateTime) {
      // If updating clock-out and clock-in exists, calculate hours
      const existingClockIn = new Date(String(existing[0].clock_in_time).replace(' ', 'T'));
      if (isNaN(existingClockIn.getTime()) || isNaN(clockOutDateTime.getTime())) {
        return res.status(400).json({ error: 'Invalid time format for calculation' });
      }
      if (clockOutDateTime < existingClockIn) {
        return res.status(400).json({ error: 'Clock-out time must be after clock-in time' });
      }
      const totalTimeMs = clockOutDateTime - existingClockIn;
      
      // Get total break time
      const [breaks] = await db.query(
        `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
         FROM work_sessions
         WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
        [existing[0].id]
      );
      const totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
      const totalBreakMs = totalBreakMinutes * 60 * 1000;
      totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);
    } else if (clockInDateTime && existing.length > 0 && existing[0].clock_out_time) {
      // If updating clock-in and clock-out exists, calculate hours
      const existingClockOut = new Date(String(existing[0].clock_out_time).replace(' ', 'T'));
      if (isNaN(existingClockOut.getTime()) || isNaN(clockInDateTime.getTime())) {
        return res.status(400).json({ error: 'Invalid time format for calculation' });
      }
      if (existingClockOut < clockInDateTime) {
        return res.status(400).json({ error: 'Clock-in time must be before clock-out time' });
      }
      const totalTimeMs = existingClockOut - clockInDateTime;
      
      // Get total break time
      const [breaks] = await db.query(
        `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
         FROM work_sessions
         WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
        [existing[0].id]
      );
      const totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
      const totalBreakMs = totalBreakMinutes * 60 * 1000;
      totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);
    }

    if (existing.length > 0) {
      // Update existing record
      const updateFields = [];
      const updateValues = [];

      if (clock_in_time !== undefined) {
        updateFields.push('clock_in_time = ?');
        updateValues.push(clockInTimeStr); // Store as string
        if (location) {
          updateFields.push('clock_in_location = ?');
          updateValues.push(location);
        }
      }

      if (clock_out_time !== undefined) {
        updateFields.push('clock_out_time = ?');
        updateValues.push(clockOutTimeStr); // Store as string
        // When admin manually edits clockout, mark it as manual (not automatic)
        updateFields.push('is_automatic_clockout = ?');
        updateValues.push(false);
        if (location) {
          updateFields.push('clock_out_location = ?');
          updateValues.push(location);
        }
      }

      if (totalHours !== null) {
        updateFields.push('total_hours = ?');
        updateValues.push(totalHours.toFixed(2));
      }

      // Set status to present if clock-in exists
      if (clockInDateTime) {
        updateFields.push('status = ?');
        updateValues.push('present');
      }

      updateValues.push(existing[0].id);

      const updateQuery = `UPDATE attendance SET ${updateFields.join(', ')} WHERE id = ?`;
      await db.query(updateQuery, updateValues);

      console.log(`✅ Attendance updated successfully for ${staff[0].full_name}`);
      console.log(`   - Date: ${attendance_date}`);
      console.log(`   - Clock In (stored as string): ${clockInTimeStr || 'Not changed'}`);
      console.log(`   - Clock Out (stored as string): ${clockOutTimeStr || 'Not changed'}`);
      console.log(`   - Total Hours: ${totalHours ? totalHours.toFixed(2) : 'N/A'}`);
      console.log('========================================\n');

      res.json({ 
        message: 'Attendance updated successfully',
        attendance: {
          id: existing[0].id,
          staff_id,
          branch_id,
          attendance_date,
          clock_in_time: clockInTimeStr || existing[0].clock_in_time,
          clock_out_time: clockOutTimeStr || existing[0].clock_out_time,
          total_hours: totalHours !== null ? totalHours.toFixed(2) : existing[0].total_hours
        }
      });
    } else {
      // Create new record
      const [result] = await db.query(
        `INSERT INTO attendance 
         (staff_id, branch_id, attendance_date, clock_in_time, clock_out_time, 
          clock_in_location, clock_out_location, total_hours, status) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          staff_id, 
          branch_id, 
          attendance_date, 
          clockInTimeStr, // Store as string
          clockOutTimeStr, // Store as string
          location || null,
          location || null,
          totalHours !== null ? totalHours.toFixed(2) : null,
          clockInTimeStr ? 'present' : 'absent'
        ]
      );

      // CRITICAL: Verify attendance_date matches what we're storing
      const storedDateStr = attendance_date.split('T')[0].split(' ')[0];
      console.log(`✅ Attendance created successfully for ${staff[0].full_name}`);
      console.log(`   - Attendance ID: ${result.insertId}`);
      console.log(`   - Date (attendance_date): ${storedDateStr} (EXPLICIT - never derived from clock_in_time)`);
      console.log(`   - Clock In (stored as string): ${clockInTimeStr || 'Not provided'}`);
      console.log(`   - Clock Out (stored as string): ${clockOutTimeStr || 'Not provided'}`);
      console.log(`   - Total Hours: ${totalHours ? totalHours.toFixed(2) : 'N/A'}`);
      console.log(`   ✅ VERIFIED: attendance_date (${storedDateStr}) will be stored, NOT derived from clock_in_time`);
      console.log('========================================\n');

      res.status(201).json({ 
        message: 'Attendance created successfully',
        attendance: {
          id: result.insertId,
          staff_id,
          branch_id,
          attendance_date,
          clock_in_time: clockInTimeStr,
          clock_out_time: clockOutTimeStr,
          total_hours: totalHours !== null ? totalHours.toFixed(2) : null
        }
      });
    }

  } catch (error) {
    console.error('❌ Create/update manual attendance error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get comprehensive reports for a branch
exports.getBranchReports = async (req, res) => {
  try {
    const { branch_id } = req.params;
    const { start_date, end_date } = req.query;

    console.log('\n========================================');
    console.log(`📊 FETCHING REPORTS FOR BRANCH ${branch_id}`);
    console.log(`📅 Date Range: ${start_date} to ${end_date}`);
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log(`📆 Server Date (Local): ${new Date().toISOString().split('T')[0]}`);
    console.log('========================================\n');

    // Verify branch belongs to this admin
    const [branches] = await db.query(`
      SELECT b.id, b.name, s.name as shop_name
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `, [branch_id, req.user.id]);

    if (branches.length === 0) {
      return res.status(403).json({ error: 'Access denied to this branch' });
    }

    console.log(`✅ Branch verified: ${branches[0].name} (Shop: ${branches[0].shop_name})`);
    console.log(`👤 Admin ID: ${req.user.id}\n`);

    const normalizeDateTime = (value) => {
      if (!value) return null;

      if (value instanceof Date) {
        if (Number.isNaN(value.getTime())) {
          return null;
        }
        return value.toISOString();
      }

      if (typeof value === 'string') {
        const trimmed = value.trim();
        if (!trimmed || trimmed.toLowerCase() === 'null' || trimmed === 'undefined') {
          return null;
        }

        const zeroDatePatterns = [
          '0000-00-00 00:00:00',
          '0000-00-00T00:00:00',
          '0000-00-00T00:00:00.000Z',
          '1970-01-01 00:00:00',
          '1970-01-01T00:00:00',
        ];

        if (zeroDatePatterns.includes(trimmed)) {
          return null;
        }

        const sanitized = trimmed.includes('T') ? trimmed : trimmed.replace(' ', 'T');

        const parsed = new Date(sanitized);
        if (Number.isNaN(parsed.getTime())) {
          return null;
        }
        return parsed.toISOString();
      }

      if (typeof value === 'number') {
        if (value <= 0) {
          return null;
        }
        const dateFromMillis = new Date(value);
        if (Number.isNaN(dateFromMillis.getTime())) {
          return null;
        }
        return dateFromMillis.toISOString();
      }

      return null;
    };

    const normalizeDate = (value) => {
      if (!value) return null;

      // IMPORTANT:
      // Never use toISOString() for DATE fields coming from MySQL, because that
      // converts the value to UTC and can shift the calendar day (e.g. 2025‑11‑20
      // in local time becomes 2025‑11‑19T18:30:00.000Z, which then formats as
      // 2025‑11‑19). We always want the exact stored calendar date.
      if (value instanceof Date) {
        if (Number.isNaN(value.getTime())) {
          return null;
        }

        // Use local date parts so the calendar day matches the DB value
        const year = value.getFullYear();
        const month = String(value.getMonth() + 1).padStart(2, '0');
        const day = String(value.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
      }

      if (typeof value === 'string') {
        const trimmed = value.trim();
        if (!trimmed || trimmed.toLowerCase() === 'null' || trimmed === 'undefined') {
          return null;
        }
        // If a time part exists, just drop it – the date portion is authoritative
        return trimmed.split('T')[0];
      }

      return null;
    };

    const normalizeTotalHours = (value) => {
      if (value === null || value === undefined) {
        return null;
      }
      if (typeof value === 'number') {
        return Number.isFinite(value) ? Number(value.toFixed(2)) : null;
      }
      if (typeof value === 'string') {
        const parsed = parseFloat(value);
        return Number.isNaN(parsed) ? null : Number(parsed.toFixed(2));
      }
      return null;
    };

    // Build date condition
    let dateCondition = '';
    const params = [branch_id];
    
    if (start_date && end_date) {
      dateCondition = 'AND a.attendance_date BETWEEN ? AND ?';
      params.push(start_date, end_date);
    } else if (start_date) {
      dateCondition = 'AND a.attendance_date >= ?';
      params.push(start_date);
    } else if (end_date) {
      dateCondition = 'AND a.attendance_date <= ?';
      params.push(end_date);
    }

    // Get all staff in the branch
    const [staffRaw] = await db.query(`
      SELECT id, staff_code, full_name, email, phone, position, hourly_rate, profile_image_path, created_at
      FROM staff
      WHERE branch_id = ? AND is_active = true
      ORDER BY full_name
    `, [branch_id]);

    // Decrypt staff data
    const staff = staffRaw.map(s => {
      try {
        s.full_name = s.full_name ? encryptionService.decrypt(s.full_name) : null;
        s.email = s.email ? encryptionService.decrypt(s.email) : null;
        s.phone = s.phone ? encryptionService.decrypt(s.phone) : null;
      } catch (error) {
        console.warn(`Failed to decrypt data for staff ${s.id}:`, error.message);
      }
      return s;
    });

    // Get attendance records
    const attendanceParams = [branch_id];
    let attendanceQuery = `
      SELECT 
        a.*,
        s.staff_code,
        s.full_name,
        s.position
      FROM attendance a
      JOIN staff s ON a.staff_id = s.id
      WHERE a.branch_id = ?
    `;
    
    if (start_date && end_date) {
      attendanceQuery += ' AND a.attendance_date BETWEEN ? AND ?';
      attendanceParams.push(start_date, end_date);
    } else if (start_date) {
      attendanceQuery += ' AND a.attendance_date >= ?';
      attendanceParams.push(start_date);
    } else if (end_date) {
      attendanceQuery += ' AND a.attendance_date <= ?';
      attendanceParams.push(end_date);
    }
    
    attendanceQuery += ' ORDER BY a.attendance_date DESC, a.clock_in_time DESC';

    console.log('📝 Attendance Query:');
    console.log(attendanceQuery);
    console.log('📝 Query Params:', attendanceParams);
    console.log('');

    const [attendanceRaw] = await db.query(attendanceQuery, attendanceParams);

    // Get break sessions for all attendance records
    const attendanceIds = attendanceRaw.map(r => r.id);
    let breaksMap = {};
    if (attendanceIds.length > 0) {
      const placeholders = attendanceIds.map(() => '?').join(',');
      const [breaks] = await db.query(
        `SELECT 
          attendance_id,
          start_time,
          end_time,
          duration_minutes,
          session_type
         FROM work_sessions
         WHERE attendance_id IN (${placeholders}) AND session_type = 'break'
         ORDER BY start_time ASC`,
        attendanceIds
      );
      
      // Group breaks by attendance_id
      breaks.forEach(breakRecord => {
        const attId = breakRecord.attendance_id;
        if (!breaksMap[attId]) {
          breaksMap[attId] = [];
        }
        breaksMap[attId].push({
          start_time: normalizeDateTime(breakRecord.start_time),
          end_time: normalizeDateTime(breakRecord.end_time),
          duration_minutes: breakRecord.duration_minutes || 0,
        });
      });
    }

    const attendance = attendanceRaw.map(record => {
      const normalizedClockIn = normalizeDateTime(record.clock_in_time);
      const normalizedClockOut = normalizeDateTime(record.clock_out_time);
      const normalizedCreatedAt = normalizeDateTime(record.created_at);
      const normalizedUpdatedAt = normalizeDateTime(record.updated_at);

      // Decrypt staff name in attendance record
      let decryptedName = record.full_name;
      try {
        if (record.full_name) {
          decryptedName = encryptionService.decrypt(record.full_name);
        }
      } catch (error) {
        // console.warn(`Failed to decrypt name for attendance ${record.id}:`, error.message);
      }

      // Get breaks for this attendance record
      const breaks = breaksMap[record.id] || [];
      const totalBreakMinutes = breaks.reduce((sum, b) => sum + (b.duration_minutes || 0), 0);

      return {
        ...record,
        full_name: decryptedName, // Use decrypted name
        clock_in_time: normalizedClockIn,
        clock_out_time: normalizedClockOut,
        attendance_date: normalizeDate(record.attendance_date),
        created_at: normalizedCreatedAt,
        updated_at: normalizedUpdatedAt,
        total_hours: normalizeTotalHours(record.total_hours),
        is_working_now: Boolean(normalizedClockIn && !normalizedClockOut),
        is_automatic_clockout: Boolean(record.is_automatic_clockout),
        breaks: breaks,
        total_break_minutes: totalBreakMinutes,
      };
    });

    console.log(`📋 Total attendance records found: ${attendance.length}`);
    
    // Debug: Show today's records specifically
    const today = new Date().toISOString().split('T')[0];
    const todayRecords = attendance.filter(a => a.attendance_date === today);
    console.log(`📅 Records for TODAY (${today}): ${todayRecords.length}`);
    
    // Debug: Show ongoing sessions
    const ongoingSessions = attendance.filter(a => a.is_working_now);
    console.log(`🔴 Ongoing sessions (no clock-out): ${ongoingSessions.length}`);
    
    if (ongoingSessions.length > 0) {
      console.log('\n📌 Details of ongoing sessions:');
      ongoingSessions.forEach((session, idx) => {
        console.log(`   ${idx + 1}. Staff: ${session.full_name} (ID: ${session.staff_id})`);
        console.log(`      Date: ${session.attendance_date}`);
        console.log(`      Clock In: ${session.clock_in_time}`);
        console.log(`      Clock Out: ${session.clock_out_time || 'NULL (still working)'}`);
        console.log(`      Status: ${session.status}`);
        console.log('');
      });
    } else {
      console.log('⚠️  No ongoing sessions found!\n');
    }

    // Calculate statistics per staff
    const staffStats = {};
    staff.forEach(s => {
      staffStats[s.id] = {
        ...s,
        total_hours: 0,
        session_count: 0,
        present_days: 0,
        absent_days: 0,
        late_days: 0,
        hourly_rate: s.hourly_rate || 0
      };
    });

    attendance.forEach(record => {
      if (staffStats[record.staff_id]) {
        staffStats[record.staff_id].session_count++;
        
        // Only count hours if NOT an automatic clockout
        if (record.total_hours && !record.is_automatic_clockout) {
          staffStats[record.staff_id].total_hours += record.total_hours;
        }
        
        if (record.status === 'present') {
          staffStats[record.staff_id].present_days++;
        } else if (record.status === 'absent') {
          staffStats[record.staff_id].absent_days++;
        } else if (record.status === 'late') {
          staffStats[record.staff_id].late_days++;
        }
      }
    });

    // Overall summary
    // Exclude automatic clockouts from total hours calculation
    const summary = {
      branch: branches[0],
      total_staff: staff.length,
      total_attendance_records: attendance.length,
      total_hours_worked: attendance.reduce((sum, a) => {
        // Only count hours if NOT an automatic clockout
        if (a.is_automatic_clockout) return sum;
        return sum + (a.total_hours || 0);
      }, 0),
      average_hours_per_day: 0,
      present_count: attendance.filter(a => a.status === 'present').length,
      absent_count: attendance.filter(a => a.status === 'absent').length,
      late_count: attendance.filter(a => a.status === 'late').length
    };

    if (summary.total_attendance_records > 0) {
      summary.average_hours_per_day = summary.total_hours_worked / summary.total_attendance_records;
    }

    res.json({
      branch: branches[0],
      staff: Object.values(staffStats),
      attendance,
      summary,
      date_range: {
        start: start_date || null,
        end: end_date || null
      }
    });

  } catch (error) {
    console.error('Get branch reports error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get breaks for an attendance record
exports.getBreaks = async (req, res) => {
  try {
    const { attendance_id } = req.params;

    const [breaks] = await db.query(
      `SELECT id, start_time, end_time, duration_minutes
       FROM work_sessions
       WHERE attendance_id = ? AND session_type = 'break'
       ORDER BY start_time ASC`,
      [attendance_id]
    );

    res.json({ breaks });
  } catch (error) {
    console.error('Get breaks error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Add break session
exports.addBreak = async (req, res) => {
  try {
    const { attendance_id, staff_id, start_time, end_time } = req.body;

    if (!attendance_id || !staff_id || !start_time) {
      return res.status(400).json({ error: 'Attendance ID, staff ID, and start time are required' });
    }

    // Verify attendance record exists and belongs to admin
    const [attendance] = await db.query(
      `SELECT a.id, a.staff_id, a.attendance_date
       FROM attendance a
       JOIN staff s ON a.staff_id = s.id
       JOIN branches b ON s.branch_id = b.id
       JOIN shops sh ON b.shop_id = sh.id
       WHERE a.id = ? AND sh.admin_id = ?`,
      [attendance_id, req.user.id]
    );

    if (attendance.length === 0) {
      return res.status(403).json({ error: 'Access denied or attendance record not found' });
    }

    const startDateTime = new Date(start_time);
    let endDateTime = end_time ? new Date(end_time) : null;
    let durationMinutes = null;

    if (endDateTime) {
      if (endDateTime < startDateTime) {
        return res.status(400).json({ error: 'End time must be after start time' });
      }
      durationMinutes = Math.round((endDateTime - startDateTime) / (1000 * 60));
    }

    const [result] = await db.query(
      `INSERT INTO work_sessions (attendance_id, staff_id, start_time, end_time, duration_minutes, session_type)
       VALUES (?, ?, ?, ?, ?, 'break')`,
      [attendance_id, staff_id, startDateTime, endDateTime, durationMinutes]
    );

    // Recalculate total hours for the attendance record
    await _recalculateAttendanceHours(attendance_id);

    res.status(201).json({
      message: 'Break added successfully',
      break_session: {
        id: result.insertId,
        attendance_id,
        start_time: startDateTime.toISOString(),
        end_time: endDateTime?.toISOString() || null,
        duration_minutes: durationMinutes,
      }
    });
  } catch (error) {
    console.error('Add break error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Update break session
exports.updateBreak = async (req, res) => {
  try {
    const { break_id } = req.params;
    const { start_time, end_time } = req.body;

    if (!start_time) {
      return res.status(400).json({ error: 'Start time is required' });
    }

    // Verify break belongs to admin
    const [breaks] = await db.query(
      `SELECT ws.id, ws.attendance_id, ws.staff_id
       FROM work_sessions ws
       JOIN attendance a ON ws.attendance_id = a.id
       JOIN staff s ON a.staff_id = s.id
       JOIN branches b ON s.branch_id = b.id
       JOIN shops sh ON b.shop_id = sh.id
       WHERE ws.id = ? AND sh.admin_id = ? AND ws.session_type = 'break'`,
      [break_id, req.user.id]
    );

    if (breaks.length === 0) {
      return res.status(403).json({ error: 'Access denied or break not found' });
    }

    const attendanceId = breaks[0].attendance_id;
    const startDateTime = new Date(start_time);
    let endDateTime = end_time ? new Date(end_time) : null;
    let durationMinutes = null;

    if (endDateTime) {
      if (endDateTime < startDateTime) {
        return res.status(400).json({ error: 'End time must be after start time' });
      }
      durationMinutes = Math.round((endDateTime - startDateTime) / (1000 * 60));
    }

    await db.query(
      `UPDATE work_sessions
       SET start_time = ?, end_time = ?, duration_minutes = ?
       WHERE id = ?`,
      [startDateTime, endDateTime, durationMinutes, break_id]
    );

    // Recalculate total hours for the attendance record
    await _recalculateAttendanceHours(attendanceId);

    res.json({
      message: 'Break updated successfully',
      break_session: {
        id: break_id,
        start_time: startDateTime.toISOString(),
        end_time: endDateTime?.toISOString() || null,
        duration_minutes: durationMinutes,
      }
    });
  } catch (error) {
    console.error('Update break error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Delete break session
exports.deleteBreak = async (req, res) => {
  try {
    const { break_id } = req.params;

    // Verify break belongs to admin and get attendance_id
    const [breaks] = await db.query(
      `SELECT ws.id, ws.attendance_id
       FROM work_sessions ws
       JOIN attendance a ON ws.attendance_id = a.id
       JOIN staff s ON a.staff_id = s.id
       JOIN branches b ON s.branch_id = b.id
       JOIN shops sh ON b.shop_id = sh.id
       WHERE ws.id = ? AND sh.admin_id = ? AND ws.session_type = 'break'`,
      [break_id, req.user.id]
    );

    if (breaks.length === 0) {
      return res.status(403).json({ error: 'Access denied or break not found' });
    }

    const attendanceId = breaks[0].attendance_id;

    await db.query('DELETE FROM work_sessions WHERE id = ?', [break_id]);

    // Recalculate total hours for the attendance record
    await _recalculateAttendanceHours(attendanceId);

    res.json({ message: 'Break deleted successfully' });
  } catch (error) {
    console.error('Delete break error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Helper function to recalculate attendance hours after break changes
async function _recalculateAttendanceHours(attendanceId) {
  try {
    const [attendance] = await db.query(
      `SELECT clock_in_time, clock_out_time FROM attendance WHERE id = ?`,
      [attendanceId]
    );

    if (attendance.length === 0 || !attendance[0].clock_in_time || !attendance[0].clock_out_time) {
      return; // Can't calculate if no clock out
    }

    const clockInTime = new Date(attendance[0].clock_in_time);
    const clockOutTime = new Date(attendance[0].clock_out_time);
    const totalTimeMs = clockOutTime - clockInTime;

    // Get total break time
    const [breaks] = await db.query(
      `SELECT COALESCE(SUM(duration_minutes), 0) as total_break_minutes
       FROM work_sessions
       WHERE attendance_id = ? AND session_type = 'break' AND end_time IS NOT NULL`,
      [attendanceId]
    );

    const totalBreakMinutes = breaks[0]?.total_break_minutes || 0;
    const totalBreakMs = totalBreakMinutes * 60 * 1000;
    const totalHours = (totalTimeMs - totalBreakMs) / (1000 * 60 * 60);

    await db.query(
      `UPDATE attendance SET total_hours = ? WHERE id = ?`,
      [totalHours.toFixed(2), attendanceId]
    );
  } catch (error) {
    console.error('Error recalculating attendance hours:', error);
  }
}

