const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const crypto = require('crypto');
const db = require('../config/database');
const conn = require('../services/db');
const DbHelper = require('../helpers/DbHelper');
const AppError = require('../utils/appError');
const { Hash, Compare } = require('../utils/encryption');
const { sendVerificationEmail, sendPasswordResetEmail } = require('../utils/emailService');
const getEnvConfig = require('../utils/Config');

// Generate JWT access token
const generateToken = (user) => {
  const payload = {
    id: user.id,
    email: user.email,
    role: user.role,
    full_name: user.full_name,
  };

  ['staff_id', 'branch_id', 'shop_id', 'admin_id', 'phone'].forEach((key) => {
    if (user[key] !== undefined) {
      payload[key] = user[key];
    }
  });

  return jwt.sign(
    payload,
    process.env.JWT_SECRET,
    { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
  );
};

// Generate refresh token
const generateRefreshToken = (user) => {
  const payload = {
    id: user.id,
    email: user.email,
    role: user.role,
  };

  return jwt.sign(
    payload,
    process.env.JWT_REFRESH_SECRET || process.env.JWT_SECRET + '_refresh',
    { expiresIn: process.env.JWT_REFRESH_EXPIRES_IN || '30d' }
  );
};

// Login
exports.login = async (req, res) => {
  try {
    const { email, password } = req.body;

    if (!email || !password) {
      return res.status(400).json({ error: 'Email and password are required' });
    }

    // Find user
    const [users] = await db.query(
      'SELECT * FROM users WHERE email = ? AND is_active = true',
      [email]
    );

    if (users.length === 0) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const user = users[0];

    // Verify password
    const isPasswordValid = await bcrypt.compare(password, user.password);
    if (!isPasswordValid) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Get user's shops if shop_admin
    let shops = [];
    if (user.role === 'shop_admin') {
      [shops] = await db.query(
        'SELECT id, name, address, contact_phone FROM shops WHERE admin_id = ? AND is_active = true',
        [user.id]
      );
      console.log(`Found ${shops.length} shops for user ${user.id}:`, shops);
    }

    // Generate tokens
    const token = generateToken(user);
    const refreshToken = generateRefreshToken(user);

    // Remove password from response
    delete user.password;

    // Check for active subscription
    let subscriptionData = null;
    let hasSubscription = false;
    
    try {
      const [subscriptions] = await db.query(
        `SELECT 
          us.id as subscription_id,
          us.plan_type,
          us.amount,
          us.start_date,
          us.end_date,
          us.status as subscription_status,
          us.created_at as subscription_created_at,
          sp.id as plan_id,
          sp.name as plan_name,
          sp.price as plan_price,
          sp.duration_type,
          sp.max_locations,
          sp.max_staff,
          sp.description as plan_description
        FROM user_subscriptions us
        LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
        WHERE us.user_id = ? 
          AND us.status = 'active' 
          AND us.end_date > NOW()
        ORDER BY us.created_at DESC
        LIMIT 1`,
        [user.id]
      );

      if (subscriptions && subscriptions.length > 0) {
        hasSubscription = true;
        subscriptionData = subscriptions[0];
      }
    } catch (subErr) {
      console.log('Error fetching subscription:', subErr);
      // Continue without subscription data
    }

    res.json({
      status: 'success',
      message: 'Logged in successfully.',
      data: {
        accessToken: token,
        refreshToken: refreshToken,
        user: {
          id: user.id,
          firstName: user.first_name || user.full_name?.split(' ')[0] || '',
          lastName: user.last_name || user.full_name?.split(' ').slice(1).join(' ') || '',
          fullName: user.full_name,
          email: user.email,
          shopName: user.shop_name || null,
          location: user.location || null,
          role: user.role,
          shops: shops
        },
        hasSubscription: hasSubscription,
        subscription: subscriptionData
      }
    });

  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Staff login via phone
exports.staffLogin = async (req, res) => {
  try {
    const { phone, staff_code: staffCode } = req.body;

    if (!phone) {
      return res.status(400).json({ error: 'Phone number is required' });
    }

    let query = `
      SELECT 
        st.*,
        b.name AS branch_name,
        b.address AS branch_address,
        b.contact_phone AS branch_phone,
        b.is_active AS branch_active,
        s.id AS shop_id,
        s.name AS shop_name,
        s.admin_id,
        s.address AS shop_address,
        s.contact_phone AS shop_phone
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.phone = ? AND st.is_active = true
    `;
    const params = [phone];

    if (staffCode) {
      query += ' AND st.staff_code = ?';
      params.push(staffCode);
    }

    const [staffRows] = await db.query(query, params);

    if (staffRows.length === 0) {
      return res.status(401).json({ error: 'Staff member not found' });
    }

    const staff = staffRows[0];

    const tokenPayload = {
      id: staff.id,
      email:
        staff.email ||
        `${staff.staff_code || staff.phone}@staff.faceml.local`,
      role: 'staff',
      full_name: staff.full_name,
      staff_id: staff.id,
      branch_id: staff.branch_id,
      shop_id: staff.shop_id,
      admin_id: staff.admin_id,
      phone: staff.phone,
    };

    const token = generateToken(tokenPayload);
    const refreshToken = generateRefreshToken(tokenPayload);

    const userPayload = {
      id: staff.id,
      email:
        staff.email ||
        `${staff.staff_code || staff.phone}@staff.faceml.local`,
      full_name: staff.full_name,
      role: 'staff',
      shops: null,
    };

    res.json({
      status: 'success',
      message: 'Staff login successful',
      data: {
        accessToken: token,
        refreshToken: refreshToken,
        user: userPayload,
        staff: {
          id: staff.id,
          full_name: staff.full_name,
          staff_code: staff.staff_code,
          phone: staff.phone,
          position: staff.position,
          hourly_rate: staff.hourly_rate,
          branch_id: staff.branch_id,
        },
        branch: {
          id: staff.branch_id,
          name: staff.branch_name,
          shop_id: staff.shop_id,
          address: staff.branch_address,
          contact_phone: staff.branch_phone,
          is_active: staff.branch_active,
        },
        shop: {
          id: staff.shop_id,
          name: staff.shop_name,
          address: staff.shop_address,
          contact_phone: staff.shop_phone,
        },
      }
    });
  } catch (error) {
    console.error('Staff login error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get current user profile
exports.getProfile = async (req, res) => {
  try {
    if (req.user.role === 'staff') {
      const [staffRows] = await db.query(
        `
        SELECT 
          st.id,
          st.full_name,
          st.email,
          st.phone,
          st.position,
          st.branch_id,
          b.name AS branch_name,
          b.shop_id,
          s.name AS shop_name
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ?
      `,
        [req.user.staff_id || req.user.id],
      );

      if (staffRows.length === 0) {
        return res.status(404).json({ error: 'Staff profile not found' });
      }

      const staff = staffRows[0];
      return res.json({
        user: {
          id: staff.id,
          email:
            staff.email ||
            `${staff.phone || 'staff'}@staff.faceml.local`,
          full_name: staff.full_name,
          role: 'staff',
          shops: null,
        },
        staff: {
          id: staff.id,
          full_name: staff.full_name,
          phone: staff.phone,
          position: staff.position,
          branch_id: staff.branch_id,
          branch_name: staff.branch_name,
          shop_id: staff.shop_id,
          shop_name: staff.shop_name,
        },
        hasSubscription: false, // Staff don't have subscriptions
      });
    }

    const [users] = await db.query(
      'SELECT id, email, full_name, role, created_at FROM users WHERE id = ?',
      [req.user.id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    let shops = [];
    if (users[0].role === 'shop_admin') {
      [shops] = await db.query(
        'SELECT id, name, address, contact_phone FROM shops WHERE admin_id = ? AND is_active = true',
        [req.user.id]
      );
      console.log(`Profile: Found ${shops.length} shops for user ${req.user.id}:`, shops);
    }

    // Check for active subscription
    let subscriptionData = null;
    let hasSubscription = false;
    
    try {
      const [subscriptions] = await db.query(
        `SELECT 
          us.id as subscription_id,
          us.plan_type,
          us.amount,
          us.start_date,
          us.end_date,
          us.status as subscription_status,
          us.created_at as subscription_created_at,
          sp.id as plan_id,
          sp.name as plan_name,
          sp.price as plan_price,
          sp.duration_type,
          sp.max_locations,
          sp.max_staff,
          sp.description as plan_description
        FROM user_subscriptions us
        LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
        WHERE us.user_id = ? 
          AND us.status = 'active' 
          AND us.end_date > NOW()
        ORDER BY us.created_at DESC
        LIMIT 1`,
        [req.user.id]
      );

      if (subscriptions && subscriptions.length > 0) {
        hasSubscription = true;
        subscriptionData = subscriptions[0];
      }
    } catch (subErr) {
      console.log('Error fetching subscription:', subErr);
      // Continue without subscription data
    }

    res.json({
      user: {
        ...users[0],
        shops: shops
      },
      hasSubscription: hasSubscription,
      subscription: subscriptionData
    });

  } catch (error) {
    console.error('Get profile error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Change password
exports.changePassword = async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;

    if (!currentPassword || !newPassword) {
      return res.status(400).json({ 
        error: 'Current password and new password are required' 
      });
    }

    if (newPassword.length < 6) {
      return res.status(400).json({ 
        error: 'New password must be at least 6 characters long' 
      });
    }

    // Get current user
    const [users] = await db.query('SELECT * FROM users WHERE id = ?', [req.user.id]);
    
    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Verify current password
    const isPasswordValid = await bcrypt.compare(currentPassword, users[0].password);
    if (!isPasswordValid) {
      return res.status(401).json({ error: 'Current password is incorrect' });
    }

    // Hash new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update password
    await db.query(
      'UPDATE users SET password = ? WHERE id = ?',
      [hashedPassword, req.user.id]
    );

    res.json({ message: 'Password changed successfully' });

  } catch (error) {
    console.error('Change password error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Refresh token
exports.refreshToken = async (req, res) => {
  try {
    const { refreshToken } = req.body;

    if (!refreshToken) {
      return res.status(400).json({ error: 'Refresh token is required' });
    }

    // Verify refresh token
    let decoded;
    try {
      decoded = jwt.verify(
        refreshToken,
        process.env.JWT_REFRESH_SECRET || process.env.JWT_SECRET + '_refresh'
      );
    } catch (error) {
      console.error('Token refresh error:', error);
      return res.status(401).json({ error: 'Invalid or expired refresh token' });
    }

    // Get user data based on role
    let userData;
    if (decoded.role === 'staff') {
      const [staffRows] = await db.query(
        `
        SELECT 
          st.*,
          b.shop_id,
          s.admin_id
        FROM staff st
        JOIN branches b ON st.branch_id = b.id
        JOIN shops s ON b.shop_id = s.id
        WHERE st.id = ? AND st.is_active = true
        `,
        [decoded.id]
      );

      if (staffRows.length === 0) {
        return res.status(404).json({ error: 'Staff member not found or inactive' });
      }

      const staff = staffRows[0];
      userData = {
        id: staff.id,
        email: staff.email || `${staff.staff_code || staff.phone}@staff.faceml.local`,
        role: 'staff',
        full_name: staff.full_name,
        staff_id: staff.id,
        branch_id: staff.branch_id,
        shop_id: staff.shop_id,
        admin_id: staff.admin_id,
        phone: staff.phone,
      };
    } else {
      const [users] = await db.query(
        'SELECT * FROM users WHERE id = ? AND is_active = true',
        [decoded.id]
      );

      if (users.length === 0) {
        return res.status(404).json({ error: 'User not found or inactive' });
      }

      userData = users[0];
    }

    // Generate new tokens
    const newToken = generateToken(userData);
    const newRefreshToken = generateRefreshToken(userData);

    res.json({
      message: 'Token refreshed successfully',
      token: newToken,
      refreshToken: newRefreshToken,
    });

  } catch (error) {
    console.error('Refresh token error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};
// Additional auth functions: forgotPassword, resetPassword, verifyEmail, register

// User Registration
exports.register = async (req, res, next) => {
  const {
    firstName,
    lastName,
    email,
    phone,
    shopName,
    location,
    password,
    confirmPassword,
  } = req?.body;

  try {
    // Trim and validate required fields
    const trimmedFirstName = firstName?.trim();
    const trimmedLastName = lastName?.trim();
    const trimmedEmail = email?.trim()?.toLowerCase();
    const trimmedPhone = phone?.trim();
    const trimmedShopName = shopName?.trim();
    const trimmedLocation = location?.trim();
    const trimmedPassword = password?.trim();
    const trimmedConfirmPassword = confirmPassword?.trim();

    // Validate required fields
    if (!trimmedFirstName) {
      return next(new AppError("First name is required", 400));
    }
    if (!trimmedLastName) {
      return next(new AppError("Last name is required", 400));
    }
    if (!trimmedEmail) {
      return next(new AppError("Email is required", 400));
    }
    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return next(new AppError("Please enter a valid email address", 400));
    }
    if (!trimmedShopName) {
      return next(new AppError("Business is required", 400));
    }
    if (!trimmedPhone) {
      return next(new AppError("Phone is required", 400));
    }
    if (!trimmedLocation) {
      return next(new AppError("Location is required", 400));
    }
    if (!trimmedPassword) {
      return next(new AppError("Password is required", 400));
    }
    if (!trimmedConfirmPassword) {
      return next(new AppError("Confirm password is required", 400));
    }

    // Validate password length (minimum 6 characters)
    if (trimmedPassword.length < 6) {
      return next(new AppError("Password must be at least 6 characters long", 400));
    }

    // Validate password match
    if (trimmedPassword !== trimmedConfirmPassword) {
      return next(new AppError("Password and confirm password do not match", 400));
    }

    // Check if email already exists
    const checkEmailQuery = "SELECT id FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    conn.query(
      checkEmailQuery,
      [trimmedEmail],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError("Something went wrong, Please try again", 500));
        }
        
        if (result && result.length > 0) {
          return next(new AppError("Email already exists. Please use a different email.", 400));
        }

        // Construct full_name from firstName and lastName
        const full_name = `${trimmedFirstName} ${trimmedLastName}`.trim();

        // Hash the password
        const hashedPassword = await Hash(trimmedPassword);

        // Generate verification token
        const verificationToken = crypto.randomBytes(32).toString("hex");
        const verificationTokenExpiry = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours

        // Insert user into database
        const query =
          "INSERT INTO `users` (first_name, last_name, full_name, email, phone, shop_name, location, password, email_verified, verification_token, verification_token_expiry, role) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'shop_admin')";
        
        conn.query(
          query,
          [trimmedFirstName, trimmedLastName, full_name, trimmedEmail, trimmedPhone, trimmedShopName, trimmedLocation, hashedPassword, 0, verificationToken, verificationTokenExpiry],
          async function (err, result, fields) {
            if (err) {
              // Check if error is due to duplicate email
              const isDuplicateEntry = 
                err.code === 'ER_DUP_ENTRY' || 
                err.code === 1062 ||
                err.errno === 1062 ||
                (err.message && (
                  err.message.includes('Duplicate entry') || 
                  err.message.includes('Duplicate') ||
                  err.message.toLowerCase().includes('email')
                ));
              
              if (isDuplicateEntry) {
                return next(new AppError("Email already exists. Please use a different email.", 400));
              }
              return next(new AppError(err?.message || "Something went wrong, Please try again", 500));
            } else if (result) {
              // Send verification email
              const baseUrl = getEnvConfig("FRONTEND_URL") || getEnvConfig("APP_URL") || "http://localhost:3000";
              const emailResult = await sendVerificationEmail(
                trimmedEmail,
                full_name,
                verificationToken,
                baseUrl
              );

              if (!emailResult.success) {
                console.error("Failed to send verification email:", emailResult.error);
              }

              return res.status(200).json({
                status: "success",
                message: "Registration successful! Please check your email to verify your account.",
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : User Registration ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
};

// Forgot Password
exports.forgotPassword = async (req, res, next) => {
  const { email } = req?.body;

  try {
    if (!email) {
      return res.status(400).json({
        status: "error",
        message: "Email is required",
      });
    }

    const trimmedEmail = email?.trim()?.toLowerCase();
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(trimmedEmail)) {
      return res.status(400).json({
        status: "error",
        message: "Please enter a valid email address",
      });
    }

    const query =
      "SELECT id, email, first_name, last_name, full_name, password FROM `users` WHERE `is_deleted` IS NULL AND email = ?";
    
    conn.query(
      query,
      [trimmedEmail],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError(err?.message || "Something went wrong, Please try again", 500));
        }

        // Always return success message for security
        if (result?.length > 0) {
          const user = result[0];

          if (user.password) {
            const resetToken = crypto.randomBytes(32).toString("hex");
            const resetTokenExpiry = new Date(Date.now() + 60 * 60 * 1000); // 1 hour

            const updateQuery =
              "UPDATE `users` SET password_reset_token = ?, password_reset_token_expiry = ? WHERE id = ?";
            
            conn.query(
              updateQuery,
              [resetToken, resetTokenExpiry, user.id],
              async function (updateErr, updateResult) {
                if (updateErr) {
                  console.error("Error updating password reset token:", updateErr);
                  return res.status(200).json({
                    status: "success",
                    message: "If an account with that email exists, a password reset link has been sent.",
                  });
                }

                const baseUrl = getEnvConfig("FRONTEND_URL") || getEnvConfig("APP_URL") || "http://localhost:3000";
                const emailResult = await sendPasswordResetEmail(
                  trimmedEmail,
                  user.full_name || `${user.first_name} ${user.last_name}`.trim() || "User",
                  resetToken,
                  baseUrl
                );

                if (!emailResult.success) {
                  console.error("Failed to send password reset email:", emailResult.error);
                }

                return res.status(200).json({
                  status: "success",
                  message: "If an account with that email exists, a password reset link has been sent.",
                });
              }
            );
          } else {
            return res.status(200).json({
              status: "success",
              message: "If an account with that email exists, a password reset link has been sent.",
            });
          }
        } else {
          return res.status(200).json({
            status: "success",
            message: "If an account with that email exists, a password reset link has been sent.",
          });
        }
      }
    );
  } catch (e) {
    console.log("Exception Error : Forgot Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
};

// Reset Password
exports.resetPassword = async (req, res, next) => {
  let token = req?.query?.token || req?.body?.token;
  const { password, confirmPassword } = req?.body;

  if (token) {
    token = token.trim();
    try {
      token = decodeURIComponent(token);
    } catch (e) {
      console.log("Token decode warning:", e.message);
    }
  }

  try {
    if (!token) {
      return res.status(400).json({
        status: "error",
        message: "Reset token is required",
      });
    }

    if (!password || !confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password are required",
      });
    }

    if (password !== confirmPassword) {
      return res.status(400).json({
        status: "error",
        message: "Password and confirm password do not match",
      });
    }

    if (password.length < 6) {
      return res.status(400).json({
        status: "error",
        message: "Password must be at least 6 characters long",
      });
    }

    const query =
      "SELECT id, email, password_reset_token_expiry FROM `users` WHERE `is_deleted` IS NULL AND password_reset_token = ?";
    
    conn.query(
      query,
      [token],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError(err?.message || "Something went wrong, Please try again", 500));
        }

        if (!result || result.length === 0) {
          return res.status(400).json({
            status: "error",
            message: "Invalid or expired reset token. Please request a new password reset link.",
          });
        }

        const user = result[0];
        const now = new Date();
        const expiryDate = new Date(user.password_reset_token_expiry);
        if (expiryDate < now) {
          return res.status(400).json({
            status: "error",
            message: "Reset token has expired. Please request a new password reset link.",
          });
        }

        const hashedPassword = await Hash(password);

        const updateQuery =
          "UPDATE `users` SET password = ?, password_reset_token = NULL, password_reset_token_expiry = NULL WHERE id = ? AND password_reset_token = ?";
        
        conn.query(
          updateQuery,
          [hashedPassword, user.id, token],
          async function (updateErr, updateResult) {
            if (updateErr) {
              return next(new AppError(updateErr?.message || "Something went wrong, Please try again", 500));
            }

            if (updateResult && updateResult.affectedRows > 0) {
              return res.status(200).json({
                status: "success",
                message: "Your password has been reset successfully. You can now log in with your new password.",
              });
            } else {
              return res.status(400).json({
                status: "error",
                message: "Failed to reset password. Please try again or request a new reset link.",
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Reset Password ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
};

// Verify Email
exports.verifyEmail = async (req, res, next) => {
  let token = req?.query?.token || req?.body?.token;
  
  if (token) {
    token = token.trim();
    try {
      token = decodeURIComponent(token);
    } catch (e) {
      console.log("Token decode warning:", e.message);
    }
  }

  try {
    if (!token) {
      return res.status(400).json({
        status: "error",
        message: "Verification token is required",
      });
    }

    const query =
      "SELECT id, email, email_verified, verification_token_expiry FROM `users` WHERE `is_deleted` IS NULL AND verification_token = ?";
    
    conn.query(
      query,
      [token],
      async function (err, result, fields) {
        if (err) {
          return next(new AppError(err?.message || "Something went wrong, Please try again", 500));
        }

        if (!result || result.length === 0) {
          return res.status(200).json({
            status: "success",
            message: "Email already verified. You can now log in.",
          });
        }

        const user = result[0];
        const emailVerifiedValue = Number(user.email_verified);
        const isVerified = emailVerifiedValue === 1;
        
        if (isVerified) {
          return res.status(200).json({
            status: "success",
            message: "Email already verified. You can now log in.",
          });
        }

        const now = new Date();
        const expiryDate = new Date(user.verification_token_expiry);
        if (expiryDate < now) {
          return res.status(400).json({
            status: "error",
            message: "Verification token has expired. Please request a new verification email.",
          });
        }

        const updateQuery =
          "UPDATE `users` SET email_verified = 1, verification_token = NULL, verification_token_expiry = NULL WHERE id = ? AND verification_token = ?";
        
        conn.query(
          updateQuery,
          [user.id, token],
          async function (updateErr, updateResult) {
            if (updateErr) {
              return next(new AppError(updateErr?.message || "Something went wrong, Please try again", 500));
            }
            
            if (updateResult && updateResult.affectedRows > 0) {
              return res.status(200).json({
                status: "success",
                message: "Your email address was verified successfully.",
              });
            } else {
              const checkQuery = "SELECT email_verified FROM `users` WHERE id = ?";
              conn.query(checkQuery, [user.id], function(checkErr, checkResult) {
                if (!checkErr && checkResult && checkResult.length > 0) {
                  if (checkResult[0].email_verified === 1) {
                    return res.status(200).json({
                      status: "success",
                      message: "Email is verified. You can now log in.",
                    });
                  }
                }
                return res.status(400).json({
                  status: "error",
                  message: "Verification failed. Please try again or contact support.",
                });
              });
            }
          }
        );
      }
    );
  } catch (e) {
    console.log("Exception Error : Email Verification ", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
};

