const db = require('../config/database');
const encryptionService = require('../services/encryptionService');

const fetchStaffContext = async (staffId) => {
  const [rows] = await db.query(
    `
    SELECT 
      st.id,
      st.branch_id,
      st.full_name,
      st.staff_code,
      st.phone,
      st.position,
      b.name AS branch_name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM staff st
    JOIN branches b ON st.branch_id = b.id
    JOIN shops s ON b.shop_id = s.id
    WHERE st.id = ?
  `,
    [staffId],
  );

  return rows[0];
};

const ensureStaffAccess = async (user, staffId) => {
  const context = await fetchStaffContext(staffId);

  if (!context) {
    const error = new Error('Staff member not found');
    error.status = 404;
    throw error;
  }

  if (user.role === 'staff') {
    if (user.staff_id !== context.id) {
      const error = new Error('You cannot access another staff member');
      error.status = 403;
      throw error;
    }
  } else if (user.role === 'shop_admin') {
    if (context.admin_id !== user.id) {
      const error = new Error('You do not have access to this staff member');
      error.status = 403;
      throw error;
    }
  } else {
    const error = new Error('Unauthorized role');
    error.status = 403;
    throw error;
  }

  return context;
};

const ensureBranchAccess = async (branchId, adminId) => {
  const [rows] = await db.query(
    `
    SELECT 
      b.id,
      b.name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM branches b
    JOIN shops s ON b.shop_id = s.id
    WHERE b.id = ?
  `,
    [branchId],
  );

  if (rows.length === 0) {
    const error = new Error('Branch not found');
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error('You do not have access to this branch');
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const normalizeAvailabilityRecord = (record) => ({
  id: record.id,
  staff_id: record.staff_id,
  branch_id: record.branch_id,
  date: record.availability_date,
  is_available: Boolean(record.is_available),
  start_minutes: record.start_minutes,
  end_minutes: record.end_minutes,
  notes: record.notes,
  submitted_by: record.submitted_by,
  submitted_by_id: record.submitted_by_id,
  created_at: record.created_at,
  updated_at: record.updated_at,
});

exports.getStaffAvailability = async (req, res) => {
  try {
    const staffId = Number(req.params.staff_id);
    if (Number.isNaN(staffId)) {
      return res.status(400).json({ error: 'Invalid staff ID' });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    const staffContext = await ensureStaffAccess(req.user, staffId);

    let query = `
      SELECT *
      FROM staff_availability
      WHERE staff_id = ?
    `;
    const params = [staffId];

    if (startDate) {
      query += ' AND availability_date >= ?';
      params.push(startDate);
    }

    if (endDate) {
      query += ' AND availability_date <= ?';
      params.push(endDate);
    }

    query += ' ORDER BY availability_date ASC';

    const [availability] = await db.query(query, params);

    res.json({
      staff: {
        id: staffContext.id,
        full_name: staffContext.full_name ? encryptionService.decrypt(staffContext.full_name) : null,
        staff_code: staffContext.staff_code,
        phone: staffContext.phone ? encryptionService.decrypt(staffContext.phone) : null,
        position: staffContext.position,
        branch_id: staffContext.branch_id,
        branch_name: staffContext.branch_name,
        shop_id: staffContext.shop_id,
        shop_name: staffContext.shop_name,
      },
      availability: availability.map(normalizeAvailabilityRecord),
    });
  } catch (error) {
    console.error('Get staff availability error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to fetch availability',
    });
  }
};

exports.upsertStaffAvailability = async (req, res) => {
  const connection = await db.getConnection();
  let transactionStarted = false;

  try {
    const staffId = Number(req.params.staff_id);
    if (Number.isNaN(staffId)) {
      return res.status(400).json({ error: 'Invalid staff ID' });
    }

    const { branch_id: branchIdInput, entries } = req.body;
    if (!Array.isArray(entries) || entries.length === 0) {
      return res.status(400).json({ error: 'Entries array is required' });
    }

    const staffContext = await ensureStaffAccess(req.user, staffId);
    const branchId = branchIdInput || staffContext.branch_id;

    if (branchId !== staffContext.branch_id) {
      return res
        .status(400)
        .json({ error: 'Branch mismatch for this staff member' });
    }

    const submitterRole = req.user.role === 'staff' ? 'staff' : 'shop_admin';
    const submitterId =
      req.user.role === 'staff' ? req.user.staff_id : req.user.id;

    await connection.beginTransaction();
    transactionStarted = true;

    for (const entry of entries) {
      const { date, is_available: isAvailable, start_minutes, end_minutes, notes } =
        entry;

      if (!date) {
        await connection.rollback();
        return res.status(400).json({ error: 'Each entry requires a date' });
      }

      if (
        isAvailable !== false &&
        (start_minutes === undefined || end_minutes === undefined)
      ) {
        await connection.rollback();
        return res.status(400).json({
          error:
            'Start and end minutes are required for available days',
        });
      }

      if (
        isAvailable !== false &&
        Number(start_minutes) >= Number(end_minutes)
      ) {
        await connection.rollback();
        return res
          .status(400)
          .json({ error: 'End time must be after start time' });
      }

      await connection.query(
        `
        INSERT INTO staff_availability (
          staff_id,
          branch_id,
          availability_date,
          is_available,
          start_minutes,
          end_minutes,
          notes,
          submitted_by,
          submitted_by_id
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          is_available = VALUES(is_available),
          start_minutes = VALUES(start_minutes),
          end_minutes = VALUES(end_minutes),
          notes = VALUES(notes),
          submitted_by = VALUES(submitted_by),
          submitted_by_id = VALUES(submitted_by_id),
          updated_at = CURRENT_TIMESTAMP
      `,
        [
          staffId,
          branchId,
          date,
          isAvailable === false ? false : true,
          isAvailable === false ? null : start_minutes,
          isAvailable === false ? null : end_minutes,
          notes || null,
          submitterRole,
          submitterId,
        ],
      );
    }

    await connection.commit();

    const dates = entries.map((entry) => entry.date).filter(Boolean);
    const minDate = dates.reduce(
      (min, current) => (current < min ? current : min),
      dates[0],
    );
    const maxDate = dates.reduce(
      (max, current) => (current > max ? current : max),
      dates[0],
    );

    let fetchQuery = `
      SELECT *
      FROM staff_availability
      WHERE staff_id = ?
    `;
    const fetchParams = [staffId];

    if (minDate && maxDate) {
      fetchQuery += ' AND availability_date BETWEEN ? AND ?';
      fetchParams.push(minDate, maxDate);
    }

    fetchQuery += ' ORDER BY availability_date ASC';

    const [updatedEntries] = await db.query(fetchQuery, fetchParams);

    res.status(201).json({
      message: 'Availability saved successfully',
      availability: updatedEntries.map(normalizeAvailabilityRecord),
    });
  } catch (error) {
    if (transactionStarted) {
      await connection.rollback();
    }
    console.error('Upsert availability error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to save availability',
    });
  } finally {
    connection.release();
  }
};

exports.getBranchAvailability = async (req, res) => {
  try {
    const branchId = Number(req.params.branch_id);
    if (Number.isNaN(branchId)) {
      return res.status(400).json({ error: 'Invalid branch ID' });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    const branchContext = await ensureBranchAccess(branchId, req.user.id);

    let query = `
      SELECT 
        sa.*,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position
      FROM staff_availability sa
      JOIN staff st ON sa.staff_id = st.id
      WHERE sa.branch_id = ?
    `;
    const params = [branchId];

    if (startDate) {
      query += ' AND sa.availability_date >= ?';
      params.push(startDate);
    }

    if (endDate) {
      query += ' AND sa.availability_date <= ?';
      params.push(endDate);
    }

    query += ' ORDER BY sa.availability_date ASC, st.full_name ASC';

    const [rows] = await db.query(query, params);

    res.json({
      branch: {
        id: branchContext.id,
        name: branchContext.name,
        shop_id: branchContext.shop_id,
        shop_name: branchContext.shop_name,
      },
      availability: rows.map((record) => ({
        ...normalizeAvailabilityRecord(record),
        staff: {
          id: record.staff_id,
          full_name: record.full_name ? encryptionService.decrypt(record.full_name) : null,
          staff_code: record.staff_code,
          phone: record.phone ? encryptionService.decrypt(record.phone) : null,
          position: record.position,
        },
      })),
    });
  } catch (error) {
    console.error('Get branch availability error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to fetch branch availability',
    });
  }
};

