const db = require('../config/database');

// Create branch
exports.createBranch = async (req, res) => {
  try {
    const { shop_id, name, address, contact_phone, pin } = req.body;

    if (!shop_id || !name) {
      return res.status(400).json({ error: 'Shop ID and branch name are required' });
    }

    // Verify shop belongs to this admin
    const [shops] = await db.query(
      'SELECT id FROM shops WHERE id = ? AND admin_id = ?',
      [shop_id, req.user.id]
    );

    if (shops.length === 0) {
      return res.status(403).json({ error: 'You do not have access to this shop' });
    }

    // Create branch
    const [result] = await db.query(
      'INSERT INTO branches (shop_id, name, address, contact_phone, pin) VALUES (?, ?, ?, ?, ?)',
      [shop_id, name, address || null, contact_phone || null, pin || null]
    );

    res.status(201).json({
      message: 'Branch created successfully',
      branch: {
        id: result.insertId,
        shop_id,
        name,
        address,
        contact_phone,
        pin
      }
    });

  } catch (error) {
    console.error('Create branch error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get all branches for a shop
exports.getBranchesByShop = async (req, res) => {
  try {
    const { shop_id } = req.params;

    // Verify shop belongs to this admin
    const [shops] = await db.query(
      'SELECT id FROM shops WHERE id = ? AND admin_id = ?',
      [shop_id, req.user.id]
    );

    if (shops.length === 0) {
      return res.status(403).json({ error: 'You do not have access to this shop' });
    }

    // Get branches with staff count
    const [branches] = await db.query(`
      SELECT 
        b.*,
        COUNT(DISTINCT s.id) as staff_count
      FROM branches b
      LEFT JOIN staff s ON b.id = s.branch_id AND s.is_active = true
      WHERE b.shop_id = ? AND b.is_active = true
      GROUP BY b.id
      ORDER BY b.created_at DESC
    `, [shop_id]);

    res.json({ branches });

  } catch (error) {
    console.error('Get branches error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get single branch
exports.getBranchById = async (req, res) => {
  try {
    const { id } = req.params;

    const [branches] = await db.query(`
      SELECT b.*, s.name as shop_name, s.admin_id
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ?
    `, [id]);

    if (branches.length === 0) {
      return res.status(404).json({ error: 'Branch not found' });
    }

    const branch = branches[0];

    // Verify access
    if (branch.admin_id !== req.user.id) {
      return res.status(403).json({ error: 'You do not have access to this branch' });
    }

    res.json({ branch });

  } catch (error) {
    console.error('Get branch error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Update branch
exports.updateBranch = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, address, contact_phone, is_active, pin } = req.body;

    // Verify branch belongs to this admin
    const [branches] = await db.query(`
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `, [id, req.user.id]);

    if (branches.length === 0) {
      return res.status(403).json({ error: 'You do not have access to this branch' });
    }

    const updateFields = [];
    const values = [];

    if (name !== undefined) {
      updateFields.push('name = ?');
      values.push(name);
    }
    if (address !== undefined) {
      updateFields.push('address = ?');
      values.push(address);
    }
    if (contact_phone !== undefined) {
      updateFields.push('contact_phone = ?');
      values.push(contact_phone);
    }
    if (is_active !== undefined) {
      updateFields.push('is_active = ?');
      values.push(is_active);
    }
    if (pin !== undefined) {
      updateFields.push('pin = ?');
      values.push(pin);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    values.push(id);

    await db.query(
      `UPDATE branches SET ${updateFields.join(', ')} WHERE id = ?`,
      values
    );

    res.json({ message: 'Branch updated successfully' });

  } catch (error) {
    console.error('Update branch error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Delete branch
exports.deleteBranch = async (req, res) => {
  try {
    const { id } = req.params;

    // Verify branch belongs to this admin
    const [branches] = await db.query(`
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `, [id, req.user.id]);

    if (branches.length === 0) {
      return res.status(403).json({ error: 'You do not have access to this branch' });
    }

    await db.query('DELETE FROM branches WHERE id = ?', [id]);

    res.json({ message: 'Branch deleted successfully' });

  } catch (error) {
    console.error('Delete branch error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Verify branch PIN
exports.verifyBranchPin = async (req, res) => {
  try {
    const { id } = req.params;
    const { pin } = req.body;

    if (!pin) {
      return res.status(400).json({ error: 'PIN is required' });
    }

    const [branches] = await db.query('SELECT pin FROM branches WHERE id = ?', [id]);

    if (branches.length === 0) {
      return res.status(404).json({ error: 'Branch not found' });
    }

    const branch = branches[0];

    // If branch has no PIN, allow access (or maybe require setting one? For now allow)
    if (!branch.pin) {
      return res.json({ success: true, message: 'No PIN set for this branch' });
    }

    if (branch.pin === pin) {
      res.json({ success: true, message: 'PIN verified' });
    } else {
      res.status(401).json({ error: 'Invalid PIN' });
    }

  } catch (error) {
    console.error('Verify branch PIN error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

