const db = require('../config/database');

// Get all feature flags
exports.getFeatureFlags = async (req, res) => {
  try {
    // Only superadmin can access feature flags
    if (req.user.role !== 'superadmin') {
      return res.status(403).json({ error: 'Access denied. Super admin only.' });
    }

    // Check if shop_id column exists
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);
    
    const hasShopIdColumn = columns.length > 0;
    const shopId = req.query.shop_id;
    
    let query, params = [];

    if (hasShopIdColumn) {
      query = 'SELECT feature_name, is_enabled, shop_id, updated_at FROM feature_flags';
      if (shopId) {
        query += ' WHERE shop_id = ?';
        params.push(shopId);
      } else {
        query += ' WHERE shop_id IS NULL';
      }
    } else {
      // Fallback: old behavior without shop_id
      query = 'SELECT feature_name, is_enabled, updated_at FROM feature_flags';
    }

    query += ' ORDER BY feature_name';

    const [flags] = await db.query(query, params);

    // Convert to object format for easier access
    const flagsObject = {};
    flags.forEach(flag => {
      flagsObject[flag.feature_name] = {
        is_enabled: flag.is_enabled === 1 || flag.is_enabled === true,
        updated_at: flag.updated_at,
        shop_id: hasShopIdColumn ? (flag.shop_id || null) : null
      };
    });

    res.json({
      message: 'Feature flags retrieved successfully',
      feature_flags: flagsObject,
      shop_id: hasShopIdColumn ? (shopId || null) : null
    });
  } catch (error) {
    console.error('Get feature flags error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Get a specific feature flag
exports.getFeatureFlag = async (req, res) => {
  try {
    if (req.user.role !== 'superadmin') {
      return res.status(403).json({ error: 'Access denied. Super admin only.' });
    }

    // Check if shop_id column exists
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);
    
    const hasShopIdColumn = columns.length > 0;
    const { featureName } = req.params;
    const shopId = req.query.shop_id;

    let query, params = [featureName];

    if (hasShopIdColumn) {
      query = 'SELECT feature_name, is_enabled, updated_at FROM feature_flags WHERE feature_name = ? AND shop_id';
      if (shopId) {
        query += ' = ?';
        params.push(shopId);
      } else {
        query += ' IS NULL';
      }
    } else {
      // Fallback: old behavior without shop_id
      query = 'SELECT feature_name, is_enabled, updated_at FROM feature_flags WHERE feature_name = ?';
    }

    const [flags] = await db.query(query, params);

    if (flags.length === 0) {
      return res.status(404).json({ error: 'Feature flag not found' });
    }

    const flag = flags[0];
    res.json({
      feature_name: flag.feature_name,
      is_enabled: flag.is_enabled === 1 || flag.is_enabled === true,
      updated_at: flag.updated_at
    });
  } catch (error) {
    console.error('Get feature flag error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Update feature flag
exports.updateFeatureFlag = async (req, res) => {
  try {
    if (req.user.role !== 'superadmin') {
      return res.status(403).json({ error: 'Access denied. Super admin only.' });
    }

    const { featureName } = req.params;
    const { is_enabled, shop_id } = req.body;

    if (typeof is_enabled !== 'boolean') {
      return res.status(400).json({ error: 'is_enabled must be a boolean value' });
    }

    // Check if shop_id column exists in feature_flags table
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);
    
    const hasShopIdColumn = columns.length > 0;

    if (hasShopIdColumn) {
      // New behavior: support shop-specific flags
      let checkQuery = 'SELECT id FROM feature_flags WHERE feature_name = ? AND shop_id';
      let checkParams = [featureName];

      if (shop_id) {
        checkQuery += ' = ?';
        checkParams.push(shop_id);
      } else {
        checkQuery += ' IS NULL';
      }

      const [existing] = await db.query(checkQuery, checkParams);

      if (existing.length === 0) {
        // Create if doesn't exist
        await db.query(
          'INSERT INTO feature_flags (feature_name, is_enabled, shop_id) VALUES (?, ?, ?)',
          [featureName, is_enabled, shop_id || null]
        );
      } else {
        // Update existing
        let updateQuery = 'UPDATE feature_flags SET is_enabled = ? WHERE feature_name = ? AND shop_id';
        let updateParams = [is_enabled, featureName];

        if (shop_id) {
          updateQuery += ' = ?';
          updateParams.push(shop_id);
        } else {
          updateQuery += ' IS NULL';
        }

        await db.query(updateQuery, updateParams);
      }
    } else {
      // Fallback: old behavior without shop_id (global flags only)
      const [existing] = await db.query(
        'SELECT id FROM feature_flags WHERE feature_name = ?',
        [featureName]
      );

      if (existing.length === 0) {
        // Create if doesn't exist
        await db.query(
          'INSERT INTO feature_flags (feature_name, is_enabled) VALUES (?, ?)',
          [featureName, is_enabled]
        );
      } else {
        // Update existing
        await db.query(
          'UPDATE feature_flags SET is_enabled = ? WHERE feature_name = ?',
          [is_enabled, featureName]
        );
      }
    }

    res.json({
      message: 'Feature flag updated successfully',
      feature_name: featureName,
      is_enabled: is_enabled,
      shop_id: hasShopIdColumn ? (shop_id || null) : null
    });
  } catch (error) {
    console.error('Update feature flag error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Check if a feature is enabled (public endpoint, no auth required for checking)
exports.checkFeatureFlag = async (req, res) => {
  try {
    // Check if shop_id column exists
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);
    
    const hasShopIdColumn = columns.length > 0;
    const { featureName } = req.params;
    const shopId = req.query.shop_id;

    // If shopId is provided and column exists, check for shop-specific flag first
    if (hasShopIdColumn && shopId) {
      const [shopFlags] = await db.query(
        'SELECT is_enabled FROM feature_flags WHERE feature_name = ? AND shop_id = ?',
        [featureName, shopId]
      );

      if (shopFlags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled: shopFlags[0].is_enabled === 1 || shopFlags[0].is_enabled === true,
          source: 'shop'
        });
      }

      // Fallback to global flag
      const [globalFlags] = await db.query(
        'SELECT is_enabled FROM feature_flags WHERE feature_name = ? AND shop_id IS NULL',
        [featureName]
      );

      if (globalFlags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled: globalFlags[0].is_enabled === 1 || globalFlags[0].is_enabled === true,
          source: 'global'
        });
      }
    } else {
      // Fallback: old behavior without shop_id
      const [flags] = await db.query(
        'SELECT is_enabled FROM feature_flags WHERE feature_name = ?',
        [featureName]
      );

      if (flags.length > 0) {
        return res.json({
          feature_name: featureName,
          is_enabled: flags[0].is_enabled === 1 || flags[0].is_enabled === true,
          source: 'global'
        });
      }
    }

    // Default to enabled if flag doesn't exist at all
    return res.json({
      feature_name: featureName,
      is_enabled: true,
      source: 'default'
    });
  } catch (error) {
    console.error('Check feature flag error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};
