const { v4: uuidv4 } = require('uuid');
const db = require('../config/database');
const encryptionService = require('../services/encryptionService');

const ensureBranchAccess = async (branchId, adminId) => {
  const [rows] = await db.query(
    `
    SELECT 
      b.id,
      b.name,
      b.shop_id,
      s.name AS shop_name,
      s.admin_id
    FROM branches b
    JOIN shops s ON b.shop_id = s.id
    WHERE b.id = ?
  `,
    [branchId],
  );

  if (rows.length === 0) {
    const error = new Error('Branch not found');
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error('You do not have access to this branch');
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const ensureRosterEntityAccess = async (table, id, adminId) => {
  const [rows] = await db.query(
    `
    SELECT 
      entity.*,
      b.id AS branch_id,
      s.admin_id
    FROM ${table} entity
    JOIN branches b ON entity.branch_id = b.id
    JOIN shops s ON b.shop_id = s.id
    WHERE entity.id = ?
  `,
    [id],
  );

  if (rows.length === 0) {
    const error = new Error('Item not found');
    error.status = 404;
    throw error;
  }

  if (rows[0].admin_id !== adminId) {
    const error = new Error('You do not have permission to modify this item');
    error.status = 403;
    throw error;
  }

  return rows[0];
};

const formatShiftType = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  startMinutes: row.start_minutes,
  endMinutes: row.end_minutes,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const formatPosition = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const formatBreakType = (row) => ({
  id: row.id,
  branch_id: row.branch_id,
  name: row.name,
  durationMinutes: row.duration_minutes,
  created_at: row.created_at,
  updated_at: row.updated_at,
});

const getBreakTypeMap = (breakTypes) => {
  const map = new Map();
  breakTypes.forEach((item) => {
    map.set(item.id, formatBreakType(item));
  });
  return map;
};

exports.getConfig = async (req, res) => {
  try {
    const branchId = Number(req.params.branch_id);
    if (Number.isNaN(branchId)) {
      return res.status(400).json({ error: 'Invalid branch ID' });
    }

    const branch = await ensureBranchAccess(branchId, req.user.id);

    const [shiftTypes] = await db.query(
      'SELECT * FROM roster_shift_types WHERE branch_id = ? ORDER BY start_minutes ASC',
      [branchId],
    );
    const [positions] = await db.query(
      'SELECT * FROM roster_positions WHERE branch_id = ? ORDER BY name ASC',
      [branchId],
    );
    const [breakTypes] = await db.query(
      'SELECT * FROM roster_break_types WHERE branch_id = ? ORDER BY duration_minutes ASC',
      [branchId],
    );

    res.json({
      branch: {
        id: branch.id,
        name: branch.name,
        shop_id: branch.shop_id,
        shop_name: branch.shop_name,
      },
      shift_types: shiftTypes.map(formatShiftType),
      positions: positions.map(formatPosition),
      break_types: breakTypes.map(formatBreakType),
    });
  } catch (error) {
    console.error('Get roster config error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to fetch roster configuration',
    });
  }
};

exports.createShiftType = async (req, res) => {
  try {
    const branchId = Number(req.body.branch_id || req.params.branch_id);
    const { name, start_minutes: startMinutes, end_minutes: endMinutes } = req.body;

    if (
      !branchId ||
      !name ||
      startMinutes === undefined ||
      endMinutes === undefined
    ) {
      return res.status(400).json({ error: 'Name and time range are required' });
    }

    if (Number(startMinutes) >= Number(endMinutes)) {
      return res
        .status(400)
        .json({ error: 'End time must be later than start time' });
    }

    await ensureBranchAccess(branchId, req.user.id);

    const id = uuidv4();
    await db.query(
      `
      INSERT INTO roster_shift_types
      (id, branch_id, name, start_minutes, end_minutes, created_by)
      VALUES (?, ?, ?, ?, ?, ?)
    `,
      [id, branchId, name.trim(), startMinutes, endMinutes, req.user.id],
    );

    const [rows] = await db.query(
      'SELECT * FROM roster_shift_types WHERE id = ?',
      [id],
    );

    res.status(201).json({ shift_type: formatShiftType(rows[0]) });
  } catch (error) {
    console.error('Create shift type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to create shift type',
    });
  }
};

exports.updateShiftType = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, start_minutes: startMinutes, end_minutes: endMinutes } = req.body;

    const existing = await ensureRosterEntityAccess(
      'roster_shift_types',
      id,
      req.user.id,
    );

    const updateFields = [];
    const values = [];

    if (name !== undefined) {
      updateFields.push('name = ?');
      values.push(name.trim());
    }

    if (startMinutes !== undefined) {
      updateFields.push('start_minutes = ?');
      values.push(startMinutes);
    }

    if (endMinutes !== undefined) {
      updateFields.push('end_minutes = ?');
      values.push(endMinutes);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    if (
      (startMinutes !== undefined || endMinutes !== undefined) &&
      Number(
        startMinutes !== undefined ? startMinutes : existing.start_minutes,
      ) >=
        Number(endMinutes !== undefined ? endMinutes : existing.end_minutes)
    ) {
      return res
        .status(400)
        .json({ error: 'End time must be later than start time' });
    }

    values.push(id);

    await db.query(
      `UPDATE roster_shift_types SET ${updateFields.join(', ')} WHERE id = ?`,
      values,
    );

    const [rows] = await db.query(
      'SELECT * FROM roster_shift_types WHERE id = ?',
      [id],
    );

    res.json({ shift_type: formatShiftType(rows[0]) });
  } catch (error) {
    console.error('Update shift type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to update shift type',
    });
  }
};

exports.deleteShiftType = async (req, res) => {
  try {
    const { id } = req.params;
    await ensureRosterEntityAccess('roster_shift_types', id, req.user.id);
    await db.query('DELETE FROM roster_shift_types WHERE id = ?', [id]);
    res.json({ message: 'Shift type deleted' });
  } catch (error) {
    console.error('Delete shift type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to delete shift type',
    });
  }
};

exports.createPosition = async (req, res) => {
  try {
    const branchId = Number(req.body.branch_id || req.params.branch_id);
    const { name } = req.body;

    if (!branchId || !name) {
      return res
        .status(400)
        .json({ error: 'Branch and position name are required' });
    }

    await ensureBranchAccess(branchId, req.user.id);

    const id = uuidv4();
    await db.query(
      `
      INSERT INTO roster_positions
        (id, branch_id, name, created_by)
      VALUES (?, ?, ?, ?)
    `,
      [id, branchId, name.trim(), req.user.id],
    );

    const [rows] = await db.query(
      'SELECT * FROM roster_positions WHERE id = ?',
      [id],
    );

    res.status(201).json({ position: formatPosition(rows[0]) });
  } catch (error) {
    console.error('Create position error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to create position',
    });
  }
};

exports.deletePosition = async (req, res) => {
  try {
    const { id } = req.params;
    await ensureRosterEntityAccess('roster_positions', id, req.user.id);
    await db.query('DELETE FROM roster_positions WHERE id = ?', [id]);
    res.json({ message: 'Position deleted' });
  } catch (error) {
    console.error('Delete position error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to delete position',
    });
  }
};

exports.createBreakType = async (req, res) => {
  try {
    const branchId = Number(req.body.branch_id || req.params.branch_id);
    const { name, duration_minutes: durationMinutes } = req.body;

    if (!branchId || !name || durationMinutes === undefined) {
      return res.status(400).json({
        error: 'Branch, break name, and duration are required',
      });
    }

    if (Number(durationMinutes) <= 0) {
      return res
        .status(400)
        .json({ error: 'Break duration must be greater than zero' });
    }

    await ensureBranchAccess(branchId, req.user.id);

    const id = uuidv4();
    await db.query(
      `
      INSERT INTO roster_break_types
        (id, branch_id, name, duration_minutes, created_by)
      VALUES (?, ?, ?, ?, ?)
    `,
      [id, branchId, name.trim(), durationMinutes, req.user.id],
    );

    const [rows] = await db.query(
      'SELECT * FROM roster_break_types WHERE id = ?',
      [id],
    );

    res.status(201).json({ break_type: formatBreakType(rows[0]) });
  } catch (error) {
    console.error('Create break type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to create break type',
    });
  }
};

exports.updateBreakType = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, duration_minutes: durationMinutes } = req.body;

    const existing = await ensureRosterEntityAccess(
      'roster_break_types',
      id,
      req.user.id,
    );

    const updateFields = [];
    const values = [];

    if (name !== undefined) {
      const trimmed = name.trim();
      if (!trimmed) {
        return res.status(400).json({ error: 'Break name cannot be empty' });
      }
      updateFields.push('name = ?');
      values.push(trimmed);
    }

    if (durationMinutes !== undefined) {
      if (Number(durationMinutes) <= 0) {
        return res
          .status(400)
          .json({ error: 'Break duration must be greater than zero' });
      }
      updateFields.push('duration_minutes = ?');
      values.push(durationMinutes);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    values.push(id);

    await db.query(
      `UPDATE roster_break_types SET ${updateFields.join(', ')} WHERE id = ?`,
      values,
    );

    const [rows] = await db.query(
      'SELECT * FROM roster_break_types WHERE id = ?',
      [id],
    );

    res.json({ break_type: formatBreakType(rows[0]) });
  } catch (error) {
    console.error('Update break type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to update break type',
    });
  }
};

exports.deleteBreakType = async (req, res) => {
  try {
    const { id } = req.params;
    await ensureRosterEntityAccess('roster_break_types', id, req.user.id);
    await db.query('DELETE FROM roster_break_types WHERE id = ?', [id]);
    res.json({ message: 'Break type deleted' });
  } catch (error) {
    console.error('Delete break type error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to delete break type',
    });
  }
};

exports.getAssignments = async (req, res) => {
  try {
    const branchId = Number(req.params.branch_id);
    if (Number.isNaN(branchId)) {
      return res.status(400).json({ error: 'Invalid branch ID' });
    }

    const { start_date: startDate, end_date: endDate } = req.query;

    await ensureBranchAccess(branchId, req.user.id);

    let query = `
      SELECT 
        ra.id,
        ra.branch_id,
        DATE_FORMAT(ra.shift_date, '%Y-%m-%d') AS shift_date,
        ra.staff_id,
        ra.shift_type_id,
        ra.position_id,
        ra.start_minutes,
        ra.end_minutes,
        ra.break_type_ids,
        ra.notes,
        ra.created_by_type,
        ra.created_by_id,
        ra.created_at,
        ra.updated_at,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position,
        rst.name AS shift_type_name,
        rst.start_minutes AS default_start_minutes,
        rst.end_minutes AS default_end_minutes,
        rp.name AS position_name
      FROM roster_assignments ra
      JOIN staff st ON ra.staff_id = st.id
      JOIN roster_shift_types rst ON ra.shift_type_id = rst.id
      LEFT JOIN roster_positions rp ON ra.position_id = rp.id
      WHERE ra.branch_id = ?
    `;
    const params = [branchId];

    if (startDate) {
      query += ' AND ra.shift_date >= ?';
      params.push(startDate);
    }

    if (endDate) {
      query += ' AND ra.shift_date <= ?';
      params.push(endDate);
    }

    query += ' ORDER BY ra.shift_date ASC, rst.start_minutes ASC';

    const [rows] = await db.query(query, params);
    const [breakTypes] = await db.query(
      'SELECT * FROM roster_break_types WHERE branch_id = ?',
      [branchId],
    );
    const breakMap = getBreakTypeMap(breakTypes);

    const assignments = rows.map((row) => {
      let breakIds = [];
      if (row.break_type_ids) {
        try {
          const parsed =
            typeof row.break_type_ids === 'string'
              ? JSON.parse(row.break_type_ids)
              : row.break_type_ids;
          if (Array.isArray(parsed)) {
            breakIds = parsed;
          }
        } catch (parseError) {
          console.warn('Failed to parse break IDs for assignment', row.id);
        }
      }

      const breaks = breakIds
        .map((id) => breakMap.get(id))
        .filter(Boolean);

      return {
        id: row.id,
        branch_id: row.branch_id,
        shift_date: row.shift_date, // safe YYYY-MM-DD string from DATE_FORMAT
        staff: {
          id: row.staff_id,
          full_name: row.full_name ? encryptionService.decrypt(row.full_name) : null,
          staff_code: row.staff_code,
          phone: row.phone ? encryptionService.decrypt(row.phone) : null,
          position: row.position,
        },
        shift_type: {
          id: row.shift_type_id,
          name: row.shift_type_name,
          startMinutes: row.start_minutes ?? row.default_start_minutes,
          endMinutes: row.end_minutes ?? row.default_end_minutes,
        },
        position: row.position_id
          ? {
              id: row.position_id,
              name: row.position_name,
            }
          : null,
        start_minutes: row.start_minutes,
        end_minutes: row.end_minutes,
        breaks,
        notes: row.notes,
      };
    });

    res.json({ assignments });
  } catch (error) {
    console.error('Get roster assignments error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to fetch roster assignments',
    });
  }
};

exports.upsertAssignments = async (req, res) => {
  const connection = await db.getConnection();
  let transactionStarted = false;

  try {
    const branchId = Number(req.params.branch_id || req.body.branch_id);
    const {
      shift_type_id: shiftTypeId,
      date,
      assignments,
      apply_to_dates: applyToDates,
      replace_existing: replaceExisting,
      remove,
    } = req.body;

    if (!branchId || !shiftTypeId || !date) {
      return res.status(400).json({
        error: 'Branch, shift type, and primary date are required',
      });
    }

    await ensureBranchAccess(branchId, req.user.id);
    const shiftType = await ensureRosterEntityAccess(
      'roster_shift_types',
      shiftTypeId,
      req.user.id,
    );

    const dates = Array.isArray(applyToDates) && applyToDates.length > 0
      ? applyToDates
      : [date];

    if (!remove && (!Array.isArray(assignments) || assignments.length === 0)) {
      return res.status(400).json({
        error: 'At least one assignment is required when not removing shifts',
      });
    }

    await connection.beginTransaction();
    transactionStarted = true;

    const uniqueStaffIds = new Set();
    const uniquePositionIds = new Set();
    const uniqueBreakIds = new Set();

    if (!remove) {
      assignments.forEach((assignment) => {
        if (assignment.staff_id) uniqueStaffIds.add(assignment.staff_id);
        if (assignment.position_id) uniquePositionIds.add(assignment.position_id);
        if (Array.isArray(assignment.break_type_ids)) {
          assignment.break_type_ids.forEach((id) => uniqueBreakIds.add(id));
        }
      });

      if (uniqueStaffIds.size > 0) {
        const staffPlaceholders = Array.from(uniqueStaffIds)
          .map(() => '?')
          .join(',');
        const [staffRows] = await connection.query(
          `
          SELECT id FROM staff 
          WHERE branch_id = ? AND id IN (${staffPlaceholders})
        `,
          [branchId, ...uniqueStaffIds],
        );
        if (staffRows.length !== uniqueStaffIds.size) {
          throw new Error('One or more staff members do not belong to this branch');
        }
      }

      if (uniquePositionIds.size > 0) {
        const placeholders = Array.from(uniquePositionIds)
          .map(() => '?')
          .join(',');
        const [positionRows] = await connection.query(
          `
          SELECT id FROM roster_positions
          WHERE branch_id = ? AND id IN (${placeholders})
        `,
          [branchId, ...uniquePositionIds],
        );
        if (positionRows.length !== uniquePositionIds.size) {
          throw new Error('One or more positions do not belong to this branch');
        }
      }

      if (uniqueBreakIds.size > 0) {
        const placeholders = Array.from(uniqueBreakIds)
          .map(() => '?')
          .join(',');
        const [breakRows] = await connection.query(
          `
          SELECT id FROM roster_break_types
          WHERE branch_id = ? AND id IN (${placeholders})
        `,
          [branchId, ...uniqueBreakIds],
        );
        if (breakRows.length !== uniqueBreakIds.size) {
          throw new Error('One or more breaks do not belong to this branch');
        }
      }
    }

    // Fast path: remove only (no inserts)
    if (remove) {
      // Bulk delete for exact dates of this shift type in this branch
      const placeholders = dates.map(() => '?').join(', ');
      await connection.query(
        `
        DELETE FROM roster_assignments
        WHERE branch_id = ?
          AND shift_type_id = ?
          AND shift_date IN (${placeholders})
      `,
        [branchId, shiftTypeId, ...dates],
      );
    }

    for (const targetDate of dates) {
      if (replaceExisting || remove) {
        await connection.query(
          `
          DELETE FROM roster_assignments
          WHERE branch_id = ? AND shift_type_id = ? AND shift_date = ?
        `,
          [branchId, shiftTypeId, targetDate],
        );
      }

      if (remove) {
        continue;
      }

      for (const assignment of assignments) {
        if (!assignment.staff_id) {
          await connection.rollback();
          return res
            .status(400)
            .json({ error: 'Each assignment requires a staff member' });
        }

        if (
          assignment.start_minutes === undefined ||
          assignment.end_minutes === undefined ||
          Number(assignment.start_minutes) >= Number(assignment.end_minutes)
        ) {
          await connection.rollback();
          return res.status(400).json({
            error: 'Each assignment requires a valid start and end time',
          });
        }

        await connection.query(
          `
          INSERT INTO roster_assignments
            (branch_id, shift_date, staff_id, shift_type_id, position_id, start_minutes, end_minutes, break_type_ids, notes, created_by_type, created_by_id)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `,
          [
            branchId,
            targetDate,
            assignment.staff_id,
            shiftTypeId,
            assignment.position_id || null,
            assignment.start_minutes,
            assignment.end_minutes,
            assignment.break_type_ids
              ? JSON.stringify(assignment.break_type_ids)
              : JSON.stringify([]),
            assignment.notes || null,
            req.user.role,
            req.user.role === 'staff' ? req.user.staff_id : req.user.id,
          ],
        );
      }
    }

    await connection.commit();

    const minDate = dates.reduce(
      (min, current) => (current < min ? current : min),
      dates[0],
    );
    const maxDate = dates.reduce(
      (max, current) => (current > max ? current : max),
      dates[0],
    );

    let fetchQuery = `
      SELECT 
        ra.*,
        st.full_name,
        st.staff_code,
        st.phone,
        st.position,
        rst.name AS shift_type_name,
        rst.start_minutes AS default_start_minutes,
        rst.end_minutes AS default_end_minutes,
        rp.name AS position_name
      FROM roster_assignments ra
      JOIN staff st ON ra.staff_id = st.id
      JOIN roster_shift_types rst ON ra.shift_type_id = rst.id
      LEFT JOIN roster_positions rp ON ra.position_id = rp.id
      WHERE ra.branch_id = ? AND ra.shift_type_id = ?
        AND ra.shift_date BETWEEN ? AND ?
      ORDER BY ra.shift_date ASC, rst.start_minutes ASC
    `;

    const [updatedRows] = await db.query(fetchQuery, [
      branchId,
      shiftTypeId,
      minDate,
      maxDate,
    ]);

    const [breakTypes] = await db.query(
      'SELECT * FROM roster_break_types WHERE branch_id = ?',
      [branchId],
    );
    const breakMap = getBreakTypeMap(breakTypes);

    const updatedAssignments = updatedRows.map((row) => {
      let breakIds = [];
      if (row.break_type_ids) {
        try {
          const parsed =
            typeof row.break_type_ids === 'string'
              ? JSON.parse(row.break_type_ids)
              : row.break_type_ids;
          if (Array.isArray(parsed)) {
            breakIds = parsed;
          }
        } catch (parseError) {
          console.warn('Failed to parse break IDs for assignment', row.id);
        }
      }

      const breaks = breakIds
        .map((id) => breakMap.get(id))
        .filter(Boolean);

      return {
        id: row.id,
        branch_id: row.branch_id,
        shift_date: row.shift_date,
        staff: {
          id: row.staff_id,
          full_name: row.full_name ? encryptionService.decrypt(row.full_name) : null,
          staff_code: row.staff_code,
          phone: row.phone ? encryptionService.decrypt(row.phone) : null,
          position: row.position,
        },
        shift_type: {
          id: row.shift_type_id,
          name: row.shift_type_name,
          startMinutes: row.start_minutes ?? row.default_start_minutes,
          endMinutes: row.end_minutes ?? row.default_end_minutes,
        },
        position: row.position_id
          ? {
              id: row.position_id,
              name: row.position_name,
            }
          : null,
        start_minutes: row.start_minutes,
        end_minutes: row.end_minutes,
        breaks,
        notes: row.notes,
      };
    });

    res.status(201).json({
      message: remove ? 'Shift assignments removed' : 'Shift assignments saved',
      assignments: updatedAssignments,
    });
  } catch (error) {
    if (transactionStarted) {
      await connection.rollback();
    }
    console.error('Upsert roster assignments error:', error);
    res.status(error.status || 500).json({
      error: error.message || 'Failed to save roster assignments',
    });
  } finally {
    connection.release();
  }
};

