const bcrypt = require('bcryptjs');
const db = require('../config/database');

// Create shop admin (superadmin only)
exports.createShopAdmin = async (req, res) => {
  const connection = await db.getConnection();
  
  try {
    let { email, password, full_name, shop_name, shop_address, contact_phone } = req.body;

    // Validation
    if (!password || !full_name) {
      return res.status(400).json({ 
        error: 'Password and full name are required' 
      });
    }

    await connection.beginTransaction();

    // Auto-generate email if not provided
    if (!email || email.trim() === '') {
      const timestamp = Date.now();
      const namePart = full_name.toLowerCase().replace(/\s+/g, '');
      email = `${namePart}_${timestamp}@faceml.local`;
    }

    // Check if email already exists
    const [existingUsers] = await connection.query(
      'SELECT id FROM users WHERE email = ?',
      [email]
    );

    if (existingUsers.length > 0) {
      await connection.rollback();
      return res.status(400).json({ error: 'Email already exists' });
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Create user
    const [userResult] = await connection.query(
      'INSERT INTO users (email, password, full_name, role) VALUES (?, ?, ?, ?)',
      [email, hashedPassword, full_name, 'shop_admin']
    );

    const userId = userResult.insertId;

    // Auto-generate shop name if not provided
    const defaultShopName = shop_name || `${full_name}'s Shop`;

    // Create default shop automatically
    const [shopResult] = await connection.query(
      'INSERT INTO shops (name, admin_id, address, contact_phone) VALUES (?, ?, ?, ?)',
      [defaultShopName, userId, shop_address || null, contact_phone || null]
    );

    const shopId = shopResult.insertId;

    // Auto-create default branch for the shop
    const defaultBranchName = 'Main Branch';
    const [branchResult] = await connection.query(
      'INSERT INTO branches (shop_id, name, address, contact_phone) VALUES (?, ?, ?, ?)',
      [shopId, defaultBranchName, shop_address || null, contact_phone || null]
    );

    console.log(`Created shop admin: ${full_name} (ID: ${userId})`);
    console.log(`Created default shop: ${defaultShopName} (ID: ${shopId})`);
    console.log(`Created default branch: ${defaultBranchName} (ID: ${branchResult.insertId})`);

    await connection.commit();

    res.status(201).json({
      message: 'Shop admin, default shop, and default branch created successfully',
      admin: {
        id: userId,
        email,
        full_name,
        shop: {
          id: shopId,
          name: defaultShopName,
          default_branch: {
            id: branchResult.insertId,
            name: defaultBranchName
          }
        }
      }
    });

  } catch (error) {
    await connection.rollback();
    console.error('Create shop admin error:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    connection.release();
  }
};

// Get all shop admins (superadmin only)
exports.getAllShopAdmins = async (req, res) => {
  try {
    // Check if pin column exists in branches table
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'branches' 
      AND COLUMN_NAME = 'pin'
    `);
    
    const hasPinColumn = columns.length > 0;
    
    // Build query based on whether pin column exists
    let query;
    if (hasPinColumn) {
      query = `
        SELECT 
          u.id,
          u.email,
          u.full_name,
          u.is_active,
          u.created_at,
          s.id as shop_id,
          s.name as shop_name,
          s.address as shop_address,
          s.contact_phone as shop_phone,
          (
            SELECT JSON_ARRAYAGG(
              JSON_OBJECT(
                'id', b.id, 
                'name', b.name, 
                'pin', b.pin
              )
            )
            FROM branches b 
            WHERE b.shop_id = s.id
          ) as branches
        FROM users u
        LEFT JOIN shops s ON u.id = s.admin_id
        WHERE u.role = 'shop_admin'
        ORDER BY u.created_at DESC
      `;
    } else {
      query = `
        SELECT 
          u.id,
          u.email,
          u.full_name,
          u.is_active,
          u.created_at,
          s.id as shop_id,
          s.name as shop_name,
          s.address as shop_address,
          s.contact_phone as shop_phone,
          (
            SELECT JSON_ARRAYAGG(
              JSON_OBJECT(
                'id', b.id, 
                'name', b.name, 
                'pin', NULL
              )
            )
            FROM branches b 
            WHERE b.shop_id = s.id
          ) as branches
        FROM users u
        LEFT JOIN shops s ON u.id = s.admin_id
        WHERE u.role = 'shop_admin'
        ORDER BY u.created_at DESC
      `;
    }

    const [shopAdmins] = await db.query(query);

    res.json({ shopAdmins });

  } catch (error) {
    console.error('Get shop admins error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Update shop admin
exports.updateShopAdmin = async (req, res) => {
  try {
    const { id } = req.params;
    const { full_name, email, is_active } = req.body;

    const updateFields = [];
    const values = [];

    if (full_name !== undefined) {
      updateFields.push('full_name = ?');
      values.push(full_name);
    }
    if (email !== undefined) {
      updateFields.push('email = ?');
      values.push(email);
    }
    if (is_active !== undefined) {
      updateFields.push('is_active = ?');
      values.push(is_active);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: 'No fields to update' });
    }

    values.push(id);

    await db.query(
      `UPDATE users SET ${updateFields.join(', ')} WHERE id = ? AND role = 'shop_admin'`,
      values
    );

    res.json({ message: 'Shop admin updated successfully' });

  } catch (error) {
    console.error('Update shop admin error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Delete shop admin
exports.deleteShopAdmin = async (req, res) => {
  try {
    const { id } = req.params;

    await db.query('DELETE FROM users WHERE id = ? AND role = ?', [id, 'shop_admin']);

    res.json({ message: 'Shop admin deleted successfully' });

  } catch (error) {
    console.error('Delete shop admin error:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

