const db = require("../config/database");
const encryptionService = require("../services/encryptionService");

const normalizeRole = (role) => {
  if (typeof role !== "string") {
    return "staff";
  }
  const trimmed = role.trim();
  return trimmed.length > 0 ? trimmed : "staff";
};

// Create staff
exports.createStaff = async (req, res) => {
  try {
    const {
      branch_id,
      staff_code,
      full_name,
      email,
      phone,
      position,
      hourly_rate,
      profile_image_path,
    } = req.body;

    if (!branch_id || !staff_code || !full_name) {
      return res.status(400).json({
        error: "Branch ID, staff code, and full name are required",
      });
    }

    // Verify branch belongs to this admin
    const [branches] = await db.query(
      `
      SELECT b.id 
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `,
      [branch_id, req.user.id],
    );

    if (branches.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this branch" });
    }

    // Check if staff code already exists
    const [existingStaff] = await db.query(
      "SELECT id FROM staff WHERE staff_code = ?",
      [staff_code],
    );

    if (existingStaff.length > 0) {
      return res.status(400).json({ error: "Staff code already exists" });
    }

    // Encrypt sensitive data
    const encryptedFullName = encryptionService.encrypt(full_name);
    const encryptedEmail = email ? encryptionService.encrypt(email) : null;
    const encryptedPhone = phone ? encryptionService.encrypt(phone) : null;

    // Create staff
    const role = normalizeRole(position);

    const [result] = await db.query(
      `INSERT INTO staff 
       (branch_id, staff_code, full_name, email, phone, position, face_embeddings, hourly_rate, profile_image_path) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        branch_id,
        staff_code,
        encryptedFullName,
        encryptedEmail,
        encryptedPhone,
        role,
        null,
        hourly_rate !== undefined ? hourly_rate : 0,
        profile_image_path || null,
      ],
    );

    res.status(201).json({
      message: "Staff created successfully",
      staff: {
        id: result.insertId,
        branch_id,
        staff_code,
        full_name, // Return unencrypted for client
        email,
        phone,
        position: role,
        hourly_rate: hourly_rate !== undefined ? hourly_rate : 0,
      },
    });
  } catch (error) {
    console.error("Create staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

// Get staff by branch
exports.getStaffByBranch = async (req, res) => {
  try {
    const { branch_id } = req.params;

    // Verify branch belongs to this admin
    const [branches] = await db.query(
      `
      SELECT b.id, b.name, s.name as shop_name
      FROM branches b
      JOIN shops s ON b.shop_id = s.id
      WHERE b.id = ? AND s.admin_id = ?
    `,
      [branch_id, req.user.id],
    );

    if (branches.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this branch" });
    }

    // Get staff list
    const [staff] = await db.query(
      `
      SELECT 
        id,
        staff_code,
        full_name,
        email,
        phone,
        position,
        hourly_rate,
        profile_image_path,
        face_embeddings,
        is_active,
        created_at
      FROM staff
      WHERE branch_id = ?
      ORDER BY created_at DESC
    `,
      [branch_id],
    );

    const staffWithDecryptedData = staff.map((member) => {
      member.position = normalizeRole(member.position);
      
      // Decrypt sensitive fields
      try {
        member.full_name = member.full_name 
          ? encryptionService.decrypt(member.full_name) 
          : null;
        member.email = member.email 
          ? encryptionService.decrypt(member.email) 
          : null;
        member.phone = member.phone 
          ? encryptionService.decrypt(member.phone) 
          : null;
      } catch (error) {
        console.warn(`Failed to decrypt data for staff ${member.id}:`, error.message);
      }
      
      // Handle face embeddings
      let embeddingPayload = member.face_embeddings;
      if (embeddingPayload) {
        try {
          // Try to decrypt if it's encrypted
          if (typeof embeddingPayload === 'string') {
            const decrypted = encryptionService.decrypt(embeddingPayload);
            embeddingPayload = JSON.parse(decrypted);
          }
        } catch (e) {
          // If decryption fails, try parsing as regular JSON
          try {
            if (typeof embeddingPayload === "string") {
              embeddingPayload = JSON.parse(embeddingPayload);
            }
          } catch (e2) {
            console.warn(
              "Failed to parse face embeddings for staff",
              member.id,
              e2,
            );
            embeddingPayload = null;
          }
        }
      }

      if (embeddingPayload && typeof embeddingPayload === "object") {
        member.face_embeddings = embeddingPayload.image || null;
      } else {
        member.face_embeddings = null;
      }

      return member;
    });

    res.json({
      branch: branches[0],
      staff: staffWithDecryptedData,
    });
  } catch (error) {
    console.error("Get staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

// Get single staff
exports.getStaffById = async (req, res) => {
  try {
    const { id } = req.params;

    const [staff] = await db.query(
      `
      SELECT 
        st.*,
        b.name as branch_name,
        s.name as shop_name,
        s.admin_id
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ?
    `,
      [id],
    );

    if (staff.length === 0) {
      return res.status(404).json({ error: "Staff not found" });
    }

    const staffMember = staff[0];
    staffMember.position = normalizeRole(staffMember.position);

    // Verify access
    if (staffMember.admin_id !== req.user.id) {
      return res
        .status(403)
        .json({ error: "You do not have access to this staff member" });
    }

    // Decrypt sensitive fields
    try {
      staffMember.full_name = staffMember.full_name 
        ? encryptionService.decrypt(staffMember.full_name) 
        : null;
      staffMember.email = staffMember.email 
        ? encryptionService.decrypt(staffMember.email) 
        : null;
      staffMember.phone = staffMember.phone 
        ? encryptionService.decrypt(staffMember.phone) 
        : null;
    } catch (error) {
      console.warn(`Failed to decrypt data for staff ${staffMember.id}:`, error.message);
    }

    // Parse face embeddings if exists
    if (staffMember.face_embeddings) {
      let embeddingPayload = staffMember.face_embeddings;
      try {
        // Try to decrypt if it's encrypted
        if (typeof embeddingPayload === 'string') {
          const decrypted = encryptionService.decrypt(embeddingPayload);
          embeddingPayload = JSON.parse(decrypted);
        }
      } catch (e) {
        // If decryption fails, try parsing as regular JSON
        try {
          if (typeof embeddingPayload === "string") {
            embeddingPayload = JSON.parse(embeddingPayload);
          }
        } catch (e2) {
          console.warn(
            "Failed to parse face embeddings for staff",
            staffMember.id,
            e2,
          );
          embeddingPayload = null;
        }
      }

      if (embeddingPayload && typeof embeddingPayload === "object") {
        staffMember.face_embeddings = embeddingPayload.image || null;
      } else {
        staffMember.face_embeddings = null;
      }
    }

    res.json({ staff: staffMember });
  } catch (error) {
    console.error("Get staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

// Update staff
exports.updateStaff = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      full_name,
      email,
      phone,
      position,
      face_embeddings,
      is_active,
      hourly_rate,
      profile_image_path,
    } = req.body;

    // Verify staff belongs to this admin
    const [staff] = await db.query(
      `
      SELECT st.id 
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND s.admin_id = ?
    `,
      [id, req.user.id],
    );

    if (staff.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this staff member" });
    }

    const updateFields = [];
    const values = [];

    if (full_name !== undefined) {
      updateFields.push("full_name = ?");
      values.push(encryptionService.encrypt(full_name));
    }
    if (email !== undefined) {
      updateFields.push("email = ?");
      values.push(email ? encryptionService.encrypt(email) : null);
    }
    if (phone !== undefined) {
      updateFields.push("phone = ?");
      values.push(phone ? encryptionService.encrypt(phone) : null);
    }
    if (position !== undefined) {
      updateFields.push("position = ?");
      values.push(normalizeRole(position));
    }
    if (is_active !== undefined) {
      updateFields.push("is_active = ?");
      values.push(is_active);
    }
    if (hourly_rate !== undefined) {
      updateFields.push("hourly_rate = ?");
      values.push(hourly_rate);
    }
    if (profile_image_path !== undefined) {
      updateFields.push("profile_image_path = ?");
      values.push(profile_image_path);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({ error: "No fields to update" });
    }

    values.push(id);

    await db.query(
      `UPDATE staff SET ${updateFields.join(", ")} WHERE id = ?`,
      values,
    );

    res.json({ message: "Staff updated successfully" });
  } catch (error) {
    console.error("Update staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

// Delete staff
exports.deleteStaff = async (req, res) => {
  try {
    const { id } = req.params;

    // Verify staff belongs to this admin
    const [staff] = await db.query(
      `
      SELECT st.id 
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND s.admin_id = ?
    `,
      [id, req.user.id],
    );

    if (staff.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this staff member" });
    }

    await db.query("DELETE FROM staff WHERE id = ?", [id]);

    res.json({ message: "Staff deleted successfully" });
  } catch (error) {
    console.error("Delete staff error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};

// Update staff face embeddings
exports.updateFaceEmbeddings = async (req, res) => {
  try {
    const { id } = req.params;
    const { face_embedding } = req.body;

    if (!face_embedding || typeof face_embedding !== "string") {
      return res
        .status(400)
        .json({ error: "face_embedding (base64 string) is required" });
    }

    // Verify staff belongs to this admin
    const [staff] = await db.query(
      `
      SELECT st.id 
      FROM staff st
      JOIN branches b ON st.branch_id = b.id
      JOIN shops s ON b.shop_id = s.id
      WHERE st.id = ? AND s.admin_id = ?
    `,
      [id, req.user.id],
    );

    if (staff.length === 0) {
      return res
        .status(403)
        .json({ error: "You do not have access to this staff member" });
    }

    // Create the embedding object and encrypt it
    const embeddingObject = JSON.stringify({ image: face_embedding });
    const encryptedEmbedding = encryptionService.encrypt(embeddingObject);

    await db.query(
      'UPDATE staff SET face_embeddings = ? WHERE id = ?',
      [encryptedEmbedding, id],
    );

    res.json({ message: "Face embedding updated successfully" });
  } catch (error) {
    console.error("Update face embeddings error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
};
