const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);

async function cancelSubscription(req, res, next) {
  try {
    const { user_id } = req?.headers;
    const { cancelImmediately = false } = req?.body; // Option to cancel immediately or at period end

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Find user's active subscription
    const findSubscriptionQuery = `
      SELECT 
        us.id,
        us.stripe_subscription_id,
        us.status,
        us.end_date
      FROM user_subscriptions us
      WHERE us.user_id = ? 
        AND us.status = 'active'
        AND us.stripe_subscription_id IS NOT NULL
      ORDER BY us.created_at DESC
      LIMIT 1
    `;

    const subscriptionResult = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      next,
      [user_id]
    );

    if (!subscriptionResult || subscriptionResult.length === 0) {
      return res.status(404).json({
        status: "error",
        message: "No active subscription found",
      });
    }

    const subscription = subscriptionResult[0];

    // Check if subscription is already cancelled
    if (subscription.status === 'cancelled') {
      return res.status(200).json({
        status: "success",
        message: "Subscription is already cancelled",
      });
    }

    try {
      // Cancel subscription in Stripe
      let cancelledSubscription;
      
      if (cancelImmediately) {
        // Cancel immediately - subscription ends now
        cancelledSubscription = await stripe.subscriptions.cancel(subscription.stripe_subscription_id);
      } else {
        // Cancel at period end - subscription continues until current period ends
        cancelledSubscription = await stripe.subscriptions.update(subscription.stripe_subscription_id, {
          cancel_at_period_end: true,
        });
      }

      // Update database
      // Always set status to 'cancelled' when user cancels (even if at period end)
      // This way the UI can show the correct state
      let newStatus = 'cancelled';
      let endDate = subscription.end_date;

      if (!cancelImmediately && cancelledSubscription.cancel_at_period_end) {
        // Subscription will be cancelled at period end, but status is 'cancelled' to show it's deactivated
        endDate = new Date(cancelledSubscription.current_period_end * 1000).toISOString().slice(0, 19).replace('T', ' ');
      } else if (cancelImmediately) {
        // Cancelled immediately
        endDate = new Date().toISOString().slice(0, 19).replace('T', ' ');
      }

      const updateQuery = `
        UPDATE user_subscriptions 
        SET status = ?,
            end_date = ?,
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        next,
        [newStatus, endDate, subscription.id]
      );

      const message = cancelImmediately
        ? "Your subscription has been cancelled immediately. Auto-renewal has been stopped."
        : "Your subscription has been scheduled for cancellation at the end of the current billing period. You will continue to have access until then.";

      return res.status(200).json({
        status: "success",
        message: message,
        data: {
          subscriptionId: subscription.id,
          status: newStatus,
          endDate: endDate,
          cancelAtPeriodEnd: cancelledSubscription.cancel_at_period_end || false,
        },
      });
    } catch (stripeError) {
      console.error("Stripe cancellation error:", stripeError);
      
      // If subscription doesn't exist in Stripe, just update database
      if (stripeError.code === 'resource_missing') {
        const updateQuery = `
          UPDATE user_subscriptions 
          SET status = 'cancelled',
              updated_at = NOW()
          WHERE id = ?
        `;

        await DbHelper.promisifyQuery(updateQuery, conn, next, [subscription.id]);

        return res.status(200).json({
          status: "success",
          message: "Subscription cancelled (Stripe subscription not found, but database updated)",
        });
      }

      return next(new AppError(stripeError.message || "Failed to cancel subscription", 500));
    }
  } catch (e) {
    console.log("Exception Error: Cancel Subscription", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = cancelSubscription;

