const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);

async function changePlan(req, res, next) {
  try {
    const { user_id } = req?.headers;
    const { newPlanId } = req?.body;

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    if (!newPlanId) {
      return next(new AppError("New plan ID is required", 400));
    }

    // Find user's active subscription
    const findSubscriptionQuery = `
      SELECT 
        us.id,
        us.stripe_subscription_id,
        us.subscription_plan_id,
        us.status,
        us.amount,
        sp.price as current_plan_price
      FROM user_subscriptions us
      LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
      WHERE us.user_id = ? 
        AND us.status = 'active'
        AND us.stripe_subscription_id IS NOT NULL
      ORDER BY us.created_at DESC
      LIMIT 1
    `;

    const subscriptionResult = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      next,
      [user_id]
    );

    if (!subscriptionResult || subscriptionResult.length === 0) {
      return res.status(404).json({
        status: "error",
        message: "No active subscription found",
      });
    }

    const subscription = subscriptionResult[0];

    // Check if trying to change to the same plan
    const newPlanIdInt = parseInt(newPlanId, 10);
    if (subscription.subscription_plan_id === newPlanIdInt) {
      return res.status(400).json({
        status: "error",
        message: "You are already subscribed to this plan",
      });
    }

    // Get new plan details (including max_locations and max_staff)
    const planQuery = `
      SELECT id, name, price, duration_type, stripe_price_id, max_locations, max_staff
      FROM subscription_plans 
      WHERE id = ? AND status = 'active' AND is_deleted IS NULL
    `;
    const planResult = await DbHelper.promisifyQuery(
      planQuery,
      conn,
      next,
      [newPlanIdInt]
    );

    if (!planResult || planResult.length === 0) {
      return next(new AppError("New subscription plan not found", 404));
    }

    const newPlan = planResult[0];

    if (!newPlan.stripe_price_id) {
      return next(new AppError("This subscription plan is not configured for recurring payments", 400));
    }

    // Determine if this is an upgrade or downgrade based on price comparison
    const currentPrice = parseFloat(subscription.current_plan_price || subscription.amount || 0);
    const newPlanPrice = parseFloat(newPlan.price);
    const isUpgrade = newPlanPrice > currentPrice;
    
    // Upgrades apply immediately, downgrades take effect at period end
    const changeImmediately = isUpgrade;

    try {
      // Retrieve current Stripe subscription
      const currentSubscription = await stripe.subscriptions.retrieve(subscription.stripe_subscription_id);
      
      // Get current subscription's currency from the current price
      const currentPriceId = currentSubscription.items.data[0].price.id;
      const currentStripePrice = await stripe.prices.retrieve(currentPriceId);
      const currentCurrency = currentStripePrice.currency.toLowerCase();

      // Get new plan's price details
      const newStripePrice = await stripe.prices.retrieve(newPlan.stripe_price_id);
      const newPriceCurrency = newStripePrice.currency.toLowerCase();

      // If currencies don't match, create a new price with the correct currency
      let priceToUse = newPlan.stripe_price_id;
      
      if (currentCurrency !== newPriceCurrency) {
        console.log(`Currency mismatch detected: Current subscription uses ${currentCurrency}, new price uses ${newPriceCurrency}. Creating new price with ${currentCurrency}.`);
        
        // Get the product ID from the new price
        const productId = typeof newStripePrice.product === 'string' ? newStripePrice.product : newStripePrice.product.id;
        
        // Create a new price with the same currency as the current subscription
        const amountInCents = Math.round(parseFloat(newPlan.price) * 100);
        const interval = newPlan.duration_type === "year" ? "year" : "month";
        
        const newPriceWithCorrectCurrency = await stripe.prices.create({
          product: productId,
          unit_amount: amountInCents,
          currency: currentCurrency,
          recurring: {
            interval: interval,
            interval_count: 1,
          },
          metadata: {
            plan_id: newPlan.id.toString(),
            duration_type: newPlan.duration_type,
            original_price_id: newPlan.stripe_price_id,
          },
        });
        
        priceToUse = newPriceWithCorrectCurrency.id;
        console.log(`Created new price ${priceToUse} with currency ${currentCurrency} for plan ${newPlan.id}`);
      }

      // Update subscription to new plan
      const updatedSubscription = await stripe.subscriptions.update(subscription.stripe_subscription_id, {
        items: [{
          id: currentSubscription.items.data[0].id,
          price: priceToUse,
        }],
        proration_behavior: changeImmediately ? 'always_invoice' : 'create_prorations',
      });

      // Update database - but keep subscription_plan_id as current plan if not immediate change
      const startDate = new Date(updatedSubscription.current_period_start * 1000).toISOString().slice(0, 19).replace('T', ' ');
      const endDate = new Date(updatedSubscription.current_period_end * 1000).toISOString().slice(0, 19).replace('T', ' ');

      if (changeImmediately) {
        // If immediate change, update plan_id right away
        const updateQuery = `
          UPDATE user_subscriptions 
          SET subscription_plan_id = ?,
              plan_type = ?,
              amount = ?,
              start_date = ?,
              end_date = ?,
              stripe_price_id = ?,
              updated_at = NOW()
          WHERE id = ?
        `;

        await DbHelper.promisifyQuery(
          updateQuery,
          conn,
          next,
          [
            newPlan.id,
            newPlan.name,
            newPlan.price,
            startDate,
            endDate,
            priceToUse,
            subscription.id
          ]
        );
      } else {
        // If change at period end, only update dates and price_id, keep current plan_id
        // The plan_id will be updated when the period actually changes (via webhook)
        const updateQuery = `
          UPDATE user_subscriptions 
          SET start_date = ?,
              end_date = ?,
              stripe_price_id = ?,
              updated_at = NOW()
          WHERE id = ?
        `;

        await DbHelper.promisifyQuery(
          updateQuery,
          conn,
          next,
          [
            startDate,
            endDate,
            priceToUse, // Store the new price_id, but plan_id stays as current plan
            subscription.id
          ]
        );
      }

      const action = isUpgrade ? 'upgraded' : 'downgraded';
      
      // Generate appropriate message based on upgrade/downgrade
      let message;
      if (isUpgrade) {
        message = `Your subscription has been ${action} to ${newPlan.name} immediately. You now have access to higher limits.`;
      } else {
        message = `Your subscription will be ${action} to ${newPlan.name} at the end of your current billing period (${endDate}). Until then, you'll continue to enjoy your current plan's higher limits.`;
      }

      return res.status(200).json({
        status: "success",
        message: message,
        data: {
          subscriptionId: subscription.id,
          newPlanId: newPlan.id,
          newPlanName: newPlan.name,
          newPlanPrice: newPlan.price,
          startDate: startDate,
          endDate: endDate,
        },
      });
    } catch (stripeError) {
      console.error("Stripe plan change error:", stripeError);
      return next(new AppError(stripeError.message || "Failed to change subscription plan", 500));
    }
  } catch (e) {
    console.log("Exception Error: Change Plan", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = changePlan;

