const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);

async function createPaymentIntent(req, res, next) {
  try {
    if (!req?.body) return next(new AppError("No request data found", 400));
    
    const { planId } = req?.body;
    const { user_id } = req?.headers;

    if (!planId) {
      return next(new AppError("Plan ID is required", 400));
    }

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Check if user already has an active subscription
    const checkSubscriptionQuery = `
      SELECT id FROM user_subscriptions 
      WHERE user_id = ? AND status = 'active' AND end_date > NOW()
    `;
    const existingSubscription = await DbHelper.promisifyQuery(
      checkSubscriptionQuery,
      conn,
      next,
      [user_id]
    );

    if (existingSubscription && existingSubscription.length > 0) {
      return res.status(200).json({
        status: "error",
        message: "You already have an active subscription. Please wait for it to expire before subscribing to a new plan.",
        hasActiveSubscription: true,
      });
    }

    // Convert planId to integer
    const planIdInt = parseInt(planId, 10);
    if (isNaN(planIdInt)) {
      return next(new AppError("Invalid plan ID format", 400));
    }

    // Fetch plan details from database (including Stripe price ID)
    const planQuery = `
      SELECT id, name, price, duration_type, stripe_price_id, stripe_product_id
      FROM subscription_plans 
      WHERE id = ? AND status = 'active' AND is_deleted IS NULL
    `;
    const planResult = await DbHelper.promisifyQuery(
      planQuery,
      conn,
      next,
      [planIdInt]
    );

    if (!planResult || planResult.length === 0) {
      console.log("Plan not found - Plan ID:", planIdInt);
      // Check if plan exists but is inactive
      const checkPlanQuery = `
        SELECT id, status, is_deleted 
        FROM subscription_plans 
        WHERE id = ?
      `;
      const checkPlan = await DbHelper.promisifyQuery(
        checkPlanQuery,
        conn,
        next,
        [planIdInt]
      );
      
      if (checkPlan && checkPlan.length > 0) {
        const plan = checkPlan[0];
        if (plan.status !== 'active') {
          return next(new AppError(`Subscription plan is ${plan.status}. Please select an active plan.`, 400));
        }
        if (plan.is_deleted) {
          return next(new AppError("Subscription plan has been deleted", 400));
        }
      }
      
      return next(new AppError("Subscription plan not found or inactive", 404));
    }

    const plan = planResult[0];
    const amount = parseFloat(plan.price);

    // Check if Stripe price ID exists
    if (!plan.stripe_price_id) {
      return next(new AppError("This subscription plan is not configured for recurring payments. Please contact support.", 400));
    }

    // Get user email for Stripe customer
    const userQuery = `SELECT email, first_name, last_name FROM users WHERE id = ?`;
    const userResult = await DbHelper.promisifyQuery(userQuery, conn, next, [user_id]);
    
    if (!userResult || userResult.length === 0) {
      return next(new AppError("User not found", 404));
    }

    const user = userResult[0];

    // Get or create Stripe Customer
    let stripeCustomerId;
    const existingCustomerQuery = `
      SELECT stripe_customer_id FROM user_subscriptions 
      WHERE user_id = ? AND stripe_customer_id IS NOT NULL 
      ORDER BY created_at DESC LIMIT 1
    `;
    const existingCustomer = await DbHelper.promisifyQuery(
      existingCustomerQuery,
      conn,
      next,
      [user_id]
    );

    if (existingCustomer && existingCustomer.length > 0 && existingCustomer[0].stripe_customer_id) {
      stripeCustomerId = existingCustomer[0].stripe_customer_id;
    } else {
      // Create new Stripe customer
      const customer = await stripe.customers.create({
        email: user.email,
        name: `${user.first_name} ${user.last_name}`.trim(),
        metadata: {
          user_id: user_id.toString(),
        },
      });
      stripeCustomerId = customer.id;
    }

    // Create Stripe Subscription
    const subscription = await stripe.subscriptions.create({
      customer: stripeCustomerId,
      items: [
        {
          price: plan.stripe_price_id,
        },
      ],
      payment_behavior: 'default_incomplete',
      payment_settings: { save_default_payment_method: 'on_subscription' },
      expand: ['latest_invoice.payment_intent'],
      metadata: {
        user_id: user_id.toString(),
        plan_id: planIdInt.toString(),
        plan_name: plan.name,
      },
    });

    const paymentIntent = subscription.latest_invoice.payment_intent;

    // Store transaction in database
    const insertQuery = `
      INSERT INTO transactions (user_id, amount, order_id, currency, status, plan_type) 
      VALUES (?, ?, ?, ?, ?, ?)
    `;

    conn.query(
      insertQuery,
      [user_id, amount, paymentIntent.id, "USD", "INITIATED", plan.name],
      function (err, result) {
        if (err) {
          console.log("Error creating transaction:", err);
          return next(
            new AppError("Failed to create transaction record", 500)
          );
        }

        res.status(200).json({
          status: "success",
          message: "Subscription setup created",
          data: {
            clientSecret: paymentIntent.client_secret,
            subscriptionId: subscription.id,
            paymentIntentId: paymentIntent.id,
            planId: planIdInt,
            planName: plan.name,
            amount: amount,
            durationType: plan.duration_type,
          },
        });
      }
    );
  } catch (e) {
    console.log("Exception Error: Create Payment Intent", e);
    return next(
      new AppError(
        e?.message || "Something went wrong, Please try again",
        500
      )
    );
  }
}

module.exports = createPaymentIntent;

