const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const moment = require("moment");

async function createSubscription(req, res, next) {
  try {
    if (!req?.body) return next(new AppError("No request data found", 400));

    const { planId, subscriptionId, paymentIntentId } = req?.body;
    const { user_id } = req?.headers;

    if (!planId || !subscriptionId) {
      return next(
        new AppError("Plan ID and subscription ID are required", 400)
      );
    }

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Verify subscription with Stripe
    const subscription = await stripe.subscriptions.retrieve(subscriptionId, {
      expand: ['latest_invoice.payment_intent', 'customer'],
    });

    // Check if subscription belongs to the user
    if (subscription.metadata?.user_id !== user_id.toString()) {
      return next(new AppError("Subscription does not belong to this user", 403));
    }

    // Verify payment intent status
    const paymentIntent = subscription.latest_invoice.payment_intent;
    if (paymentIntent.status !== "succeeded") {
      return next(
        new AppError("Payment not completed. Please try again.", 400)
      );
    }

    // Get plan details from metadata or database
    const planIdFromMetadata = subscription.metadata?.plan_id || planId;
    const planIdInt = parseInt(planIdFromMetadata, 10);
    
    if (isNaN(planIdInt)) {
      return next(new AppError("Invalid plan ID format", 400));
    }

    const planQuery = `
      SELECT id, name, price, duration_type, stripe_price_id 
      FROM subscription_plans 
      WHERE id = ? AND status = 'active' AND is_deleted IS NULL
    `;
    const planResult = await DbHelper.promisifyQuery(
      planQuery,
      conn,
      next,
      [planIdInt]
    );

    if (!planResult || planResult.length === 0) {
      console.log("Plan not found in createSubscription - Plan ID:", planIdInt);
      return next(new AppError("Subscription plan not found", 404));
    }

    const plan = planResult[0];
    const amount = parseFloat(plan.price);

    // Get subscription dates from Stripe
    const startDate = moment.unix(subscription.current_period_start).format("YYYY-MM-DD HH:mm:ss");
    const endDate = moment.unix(subscription.current_period_end).format("YYYY-MM-DD HH:mm:ss");

    // Update transaction status if paymentIntentId is provided
    if (paymentIntentId) {
      const updateTransactionQuery = `
        UPDATE transactions 
        SET status = ? 
        WHERE order_id = ? AND user_id = ?
      `;

      await DbHelper.promisifyQuery(
        updateTransactionQuery,
        conn,
        next,
        ["COMPLETED", paymentIntentId, user_id]
      );
    }

    // Create new subscription record with Stripe subscription details
    const insertSubscriptionQuery = `
      INSERT INTO user_subscriptions (
        user_id, 
        subscription_plan_id, 
        plan_type, 
        amount, 
        start_date, 
        end_date, 
        status, 
        payment_intent_id,
        stripe_subscription_id,
        stripe_customer_id,
        stripe_price_id
      ) 
      VALUES (?, ?, ?, ?, ?, ?, 'active', ?, ?, ?, ?)
    `;

    const subscriptionResult = await DbHelper.promisifyQuery(
      insertSubscriptionQuery,
      conn,
      next,
      [
        user_id, 
        plan.id, 
        plan.name, 
        amount, 
        startDate, 
        endDate, 
        paymentIntentId || subscription.latest_invoice.payment_intent.id,
        subscription.id,
        subscription.customer.id || subscription.customer,
        plan.stripe_price_id
      ]
    );

    res.status(200).json({
      status: "success",
      message: "Subscription created successfully",
      data: {
        subscriptionId: subscriptionResult.insertId,
        stripeSubscriptionId: subscription.id,
        planId: plan.id,
        planName: plan.name,
        amount: amount,
        durationType: plan.duration_type,
        startDate: startDate,
        endDate: endDate,
      },
    });
  } catch (e) {
    console.log("Exception Error: Create Subscription", e);
    return next(
      new AppError(
        e?.message || "Something went wrong, Please try again",
        500
      )
    );
  }
}

module.exports = createSubscription;

