const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const moment = require("moment");

async function getSubscriptionHistory(req, res, next) {
  try {
    const { user_id } = req?.headers;

    if (!user_id) {
      return next(new AppError("User authentication required", 401));
    }

    // Get all subscriptions for the user
    const subscriptionsQuery = `
      SELECT 
        us.id,
        us.user_id,
        us.subscription_plan_id,
        us.plan_type,
        us.amount,
        us.start_date,
        us.end_date,
        us.status,
        us.stripe_subscription_id,
        us.stripe_price_id,
        us.created_at,
        us.updated_at,
        sp.name as plan_name,
        sp.duration_type,
        sp.max_locations,
        sp.max_staff,
        sp.stripe_price_id as current_plan_stripe_price_id
      FROM user_subscriptions us
      LEFT JOIN subscription_plans sp ON us.subscription_plan_id = sp.id
      WHERE us.user_id = ?
      ORDER BY us.created_at DESC
    `;

    const subscriptions = await DbHelper.promisifyQuery(
      subscriptionsQuery,
      conn,
      next,
      [user_id]
    );

    // Get all subscription-related transactions
    const transactionsQuery = `
      SELECT 
        t.id,
        t.user_id,
        t.amount,
        t.order_id,
        t.currency,
        t.status,
        t.plan_type,
        t.created_at,
        t.updated_at
      FROM transactions t
      WHERE t.user_id = ? 
        AND t.plan_type IS NOT NULL
        AND t.status = 'COMPLETED'
      ORDER BY t.created_at DESC
    `;

    const transactions = await DbHelper.promisifyQuery(
      transactionsQuery,
      conn,
      next,
      [user_id]
    );

    // Build history array combining subscriptions and transactions
    const history = [];

    // Check for pending downgrades - when stripe_price_id differs from current plan's price_id
    // This indicates a scheduled plan change (downgrades are scheduled, upgrades are instant)
    for (const sub of subscriptions) {
      // Check subscriptions that haven't expired (active or cancelled but still valid)
      const isSubscriptionValid = moment(sub.end_date).isAfter(moment());
      
      if (isSubscriptionValid) {
        // Check if there's a pending plan change (scheduled downgrade)
        // When a downgrade is scheduled, stripe_price_id is updated but subscription_plan_id stays the same
        // Also check if updated_at is different from created_at (meaning it was modified)
        const wasUpdated = sub.updated_at && 
                          moment(sub.updated_at).isAfter(moment(sub.created_at), 'minute');
        
        // Check if there's a pending plan change (scheduled downgrade)
        // When a downgrade is scheduled, stripe_price_id is updated but subscription_plan_id stays the same
        // We need both values to exist and be different
        const hasPendingChange = sub.stripe_price_id && 
                                 sub.current_plan_stripe_price_id && 
                                 sub.stripe_price_id !== sub.current_plan_stripe_price_id &&
                                 wasUpdated;
        
        if (hasPendingChange) {
          
          // Find the plan that matches the scheduled stripe_price_id
          const pendingPlanQuery = `
            SELECT id, name, price, duration_type
            FROM subscription_plans 
            WHERE stripe_price_id = ? AND status = 'active' AND is_deleted IS NULL
            LIMIT 1
          `;
          
          const pendingPlanResult = await DbHelper.promisifyQuery(
            pendingPlanQuery,
            conn,
            null,
            [sub.stripe_price_id]
          ).catch((err) => {
            console.error('Error fetching pending plan:', err);
            return [];
          });

          let pendingPlan = null;
          let pendingAmount = 0;

          if (pendingPlanResult && pendingPlanResult.length > 0) {
            // Found plan in database
            pendingPlan = pendingPlanResult[0];
            pendingAmount = parseFloat(pendingPlan.price) || 0;
          } else {
            // Plan not found in database - try to get it from Stripe
            // This happens when a price was created dynamically (e.g., currency mismatch)
            try {
              const stripePrice = await stripe.prices.retrieve(sub.stripe_price_id);
              
              // Check if the price has metadata with plan_id
              if (stripePrice.metadata && stripePrice.metadata.plan_id) {
                const planIdFromMetadata = parseInt(stripePrice.metadata.plan_id, 10);
                
                // Try to get plan from database using the plan_id from metadata
                const planFromMetadataQuery = `
                  SELECT id, name, price, duration_type
                  FROM subscription_plans 
                  WHERE id = ? AND status = 'active' AND is_deleted IS NULL
                  LIMIT 1
                `;
                
                const planFromMetadata = await DbHelper.promisifyQuery(
                  planFromMetadataQuery,
                  conn,
                  null,
                  [planIdFromMetadata]
                ).catch(() => []);

                if (planFromMetadata && planFromMetadata.length > 0) {
                  pendingPlan = planFromMetadata[0];
                  pendingAmount = parseFloat(pendingPlan.price) || 0;
                } else {
                  // Use Stripe price data as fallback
                  pendingAmount = (stripePrice.unit_amount || 0) / 100; // Convert from cents
                  pendingPlan = {
                    id: planIdFromMetadata || null,
                    name: stripePrice.metadata.plan_name || `Plan (${stripePrice.id})`,
                    price: pendingAmount,
                    duration_type: stripePrice.metadata.duration_type || stripePrice.recurring?.interval || 'month',
                  };
                }
              } else {
                // No metadata, use Stripe price data directly
                pendingAmount = (stripePrice.unit_amount || 0) / 100;
                pendingPlan = {
                  id: null,
                  name: `Plan (${stripePrice.id.substring(0, 20)}...)`,
                  price: pendingAmount,
                  duration_type: stripePrice.recurring?.interval || 'month',
                };
              }
            } catch (stripeError) {
              console.error(`Error fetching Stripe price ${sub.stripe_price_id}:`, stripeError);
              // Still show the downgrade request with available info
              pendingAmount = 0; // Unknown amount
              pendingPlan = {
                id: null,
                name: 'Unknown Plan',
                price: 0,
                duration_type: 'month',
              };
            }
          }

          // Now add the history entry if we have plan info
          if (pendingPlan) {
            const currentAmount = parseFloat(sub.amount) || 0;

            // Determine if it's an upgrade or downgrade
            if (pendingAmount < currentAmount || (pendingAmount === 0 && currentAmount > 0)) {
              // Scheduled downgrade - add it to history
              console.log(`Found pending downgrade: Subscription ${sub.id}, from ${sub.plan_name} ($${currentAmount}) to ${pendingPlan.name} ($${pendingAmount})`);
              history.push({
                id: `pending_downgrade_${sub.id}`,
                type: 'pending_downgrade',
                action: 'Downgrade Requested',
                plan_name: pendingPlan.name,
                plan_id: pendingPlan.id,
                amount: pendingAmount,
                date: moment(sub.updated_at).format('YYYY-MM-DD HH:mm:ss'),
                status: 'pending',
                details: {
                  current_plan: sub.plan_name || sub.plan_type,
                  current_amount: currentAmount,
                  effective_date: moment(sub.end_date).format('YYYY-MM-DD HH:mm:ss'),
                  message: `Will downgrade to ${pendingPlan.name} at the end of billing period`,
                  stripe_price_id: sub.stripe_price_id, // Include for reference
                },
              });
            } else if (pendingAmount > currentAmount) {
              // This shouldn't happen for upgrades (they're instant), but handle it anyway
              history.push({
                id: `pending_upgrade_${sub.id}`,
                type: 'pending_upgrade',
                action: 'Upgrade Scheduled',
                plan_name: pendingPlan.name,
                plan_id: pendingPlan.id,
                amount: pendingAmount,
                date: moment(sub.updated_at).format('YYYY-MM-DD HH:mm:ss'),
                status: 'pending',
                details: {
                  current_plan: sub.plan_name || sub.plan_type,
                  current_amount: currentAmount,
                  effective_date: moment(sub.end_date).format('YYYY-MM-DD HH:mm:ss'),
                  message: `Will upgrade to ${pendingPlan.name} at the end of billing period`,
                },
              });
            }
          }
        }
      }

      // Initial subscription creation
      history.push({
        id: `sub_${sub.id}`,
        type: 'subscription_created',
        action: 'Subscription Started',
        plan_name: sub.plan_name || sub.plan_type,
        plan_id: sub.subscription_plan_id,
        amount: parseFloat(sub.amount) || 0,
        date: moment(sub.created_at).format('YYYY-MM-DD HH:mm:ss'),
        status: 'completed',
        details: {
          start_date: moment(sub.start_date).format('YYYY-MM-DD HH:mm:ss'),
          end_date: moment(sub.end_date).format('YYYY-MM-DD HH:mm:ss'),
          duration_type: sub.duration_type,
          max_locations: sub.max_locations,
          max_staff: sub.max_staff,
        },
      });
    }

    // Track previous subscription to detect upgrades/downgrades
    let previousSub = null;

    // Add transaction events (renewals and payments)
    transactions.forEach((txn, index) => {
      // Find the subscription this transaction belongs to
      // Match by plan_type and find the subscription that was active at transaction time
      const relatedSub = subscriptions.find((sub) => {
        if (sub.plan_type === txn.plan_type) {
          const txnDate = moment(txn.created_at);
          const subCreatedDate = moment(sub.created_at);
          const subEndDate = moment(sub.end_date);
          // Transaction should be within subscription period or shortly after renewal
          return txnDate.isAfter(subCreatedDate, 'minute') && 
                 (txnDate.isBefore(subEndDate, 'day') || txnDate.isSame(subEndDate, 'day'));
        }
        return false;
      });

      // Determine if this is likely a renewal or initial payment
      let action = 'Payment';
      let type = 'payment';

      if (relatedSub) {
        // Check if transaction happened after subscription creation (likely renewal)
        const txnDate = moment(txn.created_at);
        const subCreatedDate = moment(relatedSub.created_at);

        if (txnDate.isAfter(subCreatedDate, 'minute')) {
          // Check if this is an upgrade or downgrade by comparing with previous subscription
          let actionType = 'Renewed';
          if (previousSub) {
            const previousAmount = parseFloat(previousSub.amount) || 0;
            const currentAmount = parseFloat(txn.amount) || 0;
            if (currentAmount > previousAmount) {
              actionType = 'Upgraded & Renewed';
              type = 'upgrade_renewal';
            } else if (currentAmount < previousAmount) {
              actionType = 'Downgraded & Renewed';
              type = 'downgrade_renewal';
            } else {
              actionType = 'Renewed';
              type = 'renewal';
            }
          } else {
            actionType = 'Renewed';
            type = 'renewal';
          }
          
          action = `Subscription ${actionType}`;
          previousSub = relatedSub;
        } else {
          // Initial payment - already covered by subscription_created event
          // Skip to avoid duplicates
          return;
        }
      } else {
        // No related subscription found, treat as standalone payment
        action = 'Payment';
        type = 'payment';
      }

      history.push({
        id: `txn_${txn.id}`,
        type: type,
        action: action,
        plan_name: txn.plan_type,
        plan_id: relatedSub?.subscription_plan_id || null,
        amount: parseFloat(txn.amount) || 0,
        date: moment(txn.created_at).format('YYYY-MM-DD HH:mm:ss'),
        status: txn.status.toLowerCase(),
        details: {
          order_id: txn.order_id,
          currency: txn.currency,
        },
      });
    });

    // Sort history by date (most recent first)
    history.sort((a, b) => {
      return moment(b.date).valueOf() - moment(a.date).valueOf();
    });

    // Remove duplicate entries (same date and type)
    const uniqueHistory = [];
    const seen = new Set();

    history.forEach((item) => {
      const key = `${item.date}_${item.type}_${item.plan_name}_${item.amount}`;
      if (!seen.has(key)) {
        seen.add(key);
        uniqueHistory.push(item);
      }
    });

    res.status(200).json({
      status: "success",
      message: "Subscription history retrieved successfully",
      data: uniqueHistory,
    });
  } catch (e) {
    console.log("Exception Error: Get Subscription History", e);
    return next(new AppError("Something went wrong, Please try again", 500));
  }
}

module.exports = getSubscriptionHistory;

