const AppError = require("../../utils/appError");
const conn = require("../../services/db");
const DbHelper = require("../../helpers/DbHelper");
const stripe = require("stripe")(process.env.STRIPE_SECRET_KEY);
const moment = require("moment");

/**
 * Handle Stripe webhook events for subscriptions
 * This handles:
 * - invoice.payment_succeeded: Subscription renewal
 * - invoice.payment_failed: Payment failure
 * - customer.subscription.updated: Subscription updated
 * - customer.subscription.deleted: Subscription cancelled
 */
async function stripeWebhook(req, res, next) {
  const sig = req.headers['stripe-signature'];
  const webhookSecret = process.env.STRIPE_WEBHOOK_SECRET;

  let event;

  try {
    // Verify webhook signature
    event = stripe.webhooks.constructEvent(req.body, sig, webhookSecret);
  } catch (err) {
    console.log(`Webhook signature verification failed: ${err.message}`);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  try {
    // Handle the event
    switch (event.type) {
      case 'invoice.payment_succeeded':
        await handleInvoicePaymentSucceeded(event.data.object);
        break;

      case 'invoice.payment_failed':
        await handleInvoicePaymentFailed(event.data.object);
        break;

      case 'customer.subscription.updated':
        await handleSubscriptionUpdated(event.data.object);
        break;

      case 'customer.subscription.deleted':
        await handleSubscriptionDeleted(event.data.object);
        break;

      default:
        console.log(`Unhandled event type: ${event.type}`);
    }

    // Return a response to acknowledge receipt of the event
    res.json({ received: true });
  } catch (error) {
    console.error('Error processing webhook:', error);
    return next(new AppError('Webhook processing failed', 500));
  }
}

/**
 * Handle successful invoice payment (subscription renewal)
 */
async function handleInvoicePaymentSucceeded(invoice) {
  try {
    const subscriptionId = invoice.subscription;
    
    if (!subscriptionId) {
      console.log('No subscription ID in invoice');
      return;
    }

    // Retrieve subscription from Stripe
    const subscription = await stripe.subscriptions.retrieve(subscriptionId);
    const userId = subscription.metadata?.user_id;

    if (!userId) {
      console.log('No user_id in subscription metadata');
      return;
    }

    // Find existing subscription in database
    const findSubscriptionQuery = `
      SELECT id, user_id, subscription_plan_id 
      FROM user_subscriptions 
      WHERE stripe_subscription_id = ? 
      ORDER BY created_at DESC 
      LIMIT 1
    `;

    const existingSubscription = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      null,
      [subscriptionId]
    );

    if (existingSubscription && existingSubscription.length > 0) {
      const sub = existingSubscription[0];
      
      // Update existing subscription with new period dates
      const startDate = moment.unix(subscription.current_period_start).format("YYYY-MM-DD HH:mm:ss");
      const endDate = moment.unix(subscription.current_period_end).format("YYYY-MM-DD HH:mm:ss");

      // Get the current active price from Stripe subscription
      const currentPriceId = subscription.items.data[0]?.price?.id;
      
      // Find the plan that matches this price_id
      let planIdToUse = sub.subscription_plan_id; // Default to current plan
      let planTypeToUse = null;
      let amountToUse = null;
      
      if (currentPriceId) {
        const planQuery = `
          SELECT id, name, price 
          FROM subscription_plans 
          WHERE stripe_price_id = ? AND status = 'active' AND is_deleted IS NULL
          LIMIT 1
        `;
        const planResult = await DbHelper.promisifyQuery(
          planQuery,
          conn,
          null,
          [currentPriceId]
        );
        
        if (planResult && planResult.length > 0) {
          const plan = planResult[0];
          planIdToUse = plan.id;
          planTypeToUse = plan.name;
          amountToUse = plan.price;
        }
      }

      // Update subscription with new period dates and plan (if it changed)
      const updateQuery = `
        UPDATE user_subscriptions 
        SET start_date = ?, 
            end_date = ?, 
            status = 'active',
            subscription_plan_id = ?,
            plan_type = ?,
            amount = ?,
            stripe_price_id = ?,
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        null,
        [
          startDate, 
          endDate, 
          planIdToUse,
          planTypeToUse || sub.plan_type,
          amountToUse || sub.amount,
          currentPriceId || sub.stripe_price_id,
          sub.id
        ]
      );

      // Create transaction record for the renewal (use the updated plan)
      if (planTypeToUse && amountToUse) {
        const insertTransactionQuery = `
          INSERT INTO transactions (user_id, amount, order_id, currency, status, plan_type) 
          VALUES (?, ?, ?, ?, ?, ?)
        `;

        await DbHelper.promisifyQuery(
          insertTransactionQuery,
          conn,
          null,
          [userId, amountToUse, invoice.payment_intent || invoice.id, "USD", "COMPLETED", planTypeToUse]
        );
      } else {
        // Fallback to original plan if price lookup failed
        const planQuery = `
          SELECT name, price FROM subscription_plans WHERE id = ?
        `;
        const planResult = await DbHelper.promisifyQuery(
          planQuery,
          conn,
          null,
          [sub.subscription_plan_id]
        );

        if (planResult && planResult.length > 0) {
          const plan = planResult[0];
          const insertTransactionQuery = `
            INSERT INTO transactions (user_id, amount, order_id, currency, status, plan_type) 
            VALUES (?, ?, ?, ?, ?, ?)
          `;

          await DbHelper.promisifyQuery(
            insertTransactionQuery,
            conn,
            null,
            [userId, plan.price, invoice.payment_intent || invoice.id, "USD", "COMPLETED", plan.name]
          );
        }
      }

      console.log(`Subscription renewed successfully for user ${userId}, subscription ${subscriptionId}`);
    } else {
      // This might be the first payment, handled by createSubscription
      console.log(`Subscription ${subscriptionId} not found in database, might be initial payment`);
    }
  } catch (error) {
    console.error('Error handling invoice.payment_succeeded:', error);
    throw error;
  }
}

/**
 * Handle failed invoice payment
 */
async function handleInvoicePaymentFailed(invoice) {
  try {
    const subscriptionId = invoice.subscription;
    
    if (!subscriptionId) {
      return;
    }

    // Find subscription in database
    const findSubscriptionQuery = `
      SELECT id, user_id 
      FROM user_subscriptions 
      WHERE stripe_subscription_id = ? 
      ORDER BY created_at DESC 
      LIMIT 1
    `;

    const existingSubscription = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      null,
      [subscriptionId]
    );

    if (existingSubscription && existingSubscription.length > 0) {
      // Update subscription status to indicate payment issue
      // Note: Don't set to 'expired' yet, Stripe will retry
      const updateQuery = `
        UPDATE user_subscriptions 
        SET status = 'past_due',
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        null,
        [existingSubscription[0].id]
      );

      console.log(`Payment failed for subscription ${subscriptionId}, status set to past_due`);
    }
  } catch (error) {
    console.error('Error handling invoice.payment_failed:', error);
    throw error;
  }
}

/**
 * Handle subscription update
 */
async function handleSubscriptionUpdated(subscription) {
  try {
    const subscriptionId = subscription.id;
    const userId = subscription.metadata?.user_id;

    if (!userId) {
      return;
    }

    // Find subscription in database
    const findSubscriptionQuery = `
      SELECT id 
      FROM user_subscriptions 
      WHERE stripe_subscription_id = ? 
      ORDER BY created_at DESC 
      LIMIT 1
    `;

    const existingSubscription = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      null,
      [subscriptionId]
    );

    if (existingSubscription && existingSubscription.length > 0) {
      const startDate = moment.unix(subscription.current_period_start).format("YYYY-MM-DD HH:mm:ss");
      const endDate = moment.unix(subscription.current_period_end).format("YYYY-MM-DD HH:mm:ss");
      
      // Determine status based on Stripe subscription status
      let status = 'active';
      if (subscription.status === 'canceled' || subscription.status === 'unpaid') {
        status = 'cancelled';
      } else if (subscription.status === 'past_due') {
        status = 'past_due';
      } else if (subscription.status === 'incomplete' || subscription.status === 'incomplete_expired') {
        status = 'expired';
      }

      const updateQuery = `
        UPDATE user_subscriptions 
        SET start_date = ?, 
            end_date = ?, 
            status = ?,
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        null,
        [startDate, endDate, status, existingSubscription[0].id]
      );

      console.log(`Subscription ${subscriptionId} updated, status: ${status}`);
    }
  } catch (error) {
    console.error('Error handling customer.subscription.updated:', error);
    throw error;
  }
}

/**
 * Handle subscription deletion/cancellation
 */
async function handleSubscriptionDeleted(subscription) {
  try {
    const subscriptionId = subscription.id;

    // Find subscription in database
    const findSubscriptionQuery = `
      SELECT id 
      FROM user_subscriptions 
      WHERE stripe_subscription_id = ? 
      ORDER BY created_at DESC 
      LIMIT 1
    `;

    const existingSubscription = await DbHelper.promisifyQuery(
      findSubscriptionQuery,
      conn,
      null,
      [subscriptionId]
    );

    if (existingSubscription && existingSubscription.length > 0) {
      const updateQuery = `
        UPDATE user_subscriptions 
        SET status = 'cancelled',
            updated_at = NOW()
        WHERE id = ?
      `;

      await DbHelper.promisifyQuery(
        updateQuery,
        conn,
        null,
        [existingSubscription[0].id]
      );

      console.log(`Subscription ${subscriptionId} cancelled`);
    }
  } catch (error) {
    console.error('Error handling customer.subscription.deleted:', error);
    throw error;
  }
}

module.exports = stripeWebhook;

