#!/bin/bash

# ============================================
# FaceML Database Dump Script
# ============================================
# This script creates a complete database dump
# including both structure and data
# ============================================

# Load environment variables if .env exists
if [ -f .env ]; then
    export $(cat .env | grep -v '^#' | xargs)
fi

# Set default values
DB_HOST=${DB_HOST:-localhost}
DB_USER=${DB_USER:-root}
DB_PASSWORD=${DB_PASSWORD:-}
DB_NAME=${DB_NAME:-faceml_db}
DB_PORT=${DB_PORT:-3306}

# Generate timestamp for filename
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
OUTPUT_FILE="faceml_database_dump_${TIMESTAMP}.sql"

echo "============================================"
echo "FaceML Database Dump"
echo "============================================"
echo "Database: $DB_NAME"
echo "Host: $DB_HOST:$DB_PORT"
echo "User: $DB_USER"
echo "Output: $OUTPUT_FILE"
echo "============================================"
echo ""

# Check if mysqldump is available
if ! command -v mysqldump &> /dev/null; then
    echo "❌ Error: mysqldump command not found"
    echo "Please install MySQL client tools:"
    echo "  macOS: brew install mysql-client"
    echo "  Ubuntu: sudo apt-get install mysql-client"
    echo "  Or use: node scripts/dumpDatabaseWithData.js"
    exit 1
fi

# Create dump with structure and data
# Using flags that don't require SUPER privileges (for shared hosting compatibility)
echo "🔄 Creating database dump..."
if [ -z "$DB_PASSWORD" ]; then
    mysqldump -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" \
        --single-transaction \
        --routines \
        --triggers \
        --events \
        --add-drop-table \
        --complete-insert \
        --extended-insert \
        --quick \
        --lock-tables=false \
        --no-tablespaces \
        --skip-set-charset \
        --skip-triggers \
        --set-gtid-purged=OFF \
        "$DB_NAME" > "$OUTPUT_FILE"
else
    mysqldump -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASSWORD" \
        --single-transaction \
        --routines \
        --triggers \
        --events \
        --add-drop-table \
        --complete-insert \
        --extended-insert \
        --quick \
        --lock-tables=false \
        --no-tablespaces \
        --skip-set-charset \
        --skip-triggers \
        --set-gtid-purged=OFF \
        "$DB_NAME" > "$OUTPUT_FILE"
fi

# Check if dump was successful
if [ $? -eq 0 ]; then
    FILE_SIZE=$(du -h "$OUTPUT_FILE" | cut -f1)
    echo ""
    echo "✅ Database dump created successfully!"
    echo "📁 File: $OUTPUT_FILE"
    echo "📊 Size: $FILE_SIZE"
    echo ""
    echo "============================================"
    echo "To import this dump on your server:"
    echo "============================================"
    echo ""
    echo "Method 1: Using mysql command line"
    echo "  mysql -u USER -p DATABASE_NAME < $OUTPUT_FILE"
    echo ""
    echo "Method 2: Using phpMyAdmin (cPanel)"
    echo "  1. Create database in cPanel"
    echo "  2. Open phpMyAdmin"
    echo "  3. Select your database"
    echo "  4. Click 'Import' tab"
    echo "  5. Choose file: $OUTPUT_FILE"
    echo "  6. Click 'Go'"
    echo ""
    echo "Method 3: Using MySQL Workbench"
    echo "  1. Connect to your server"
    echo "  2. Server > Data Import"
    echo "  3. Select 'Import from Self-Contained File'"
    echo "  4. Choose: $OUTPUT_FILE"
    echo "  5. Click 'Start Import'"
    echo ""
else
    echo ""
    echo "❌ Error: Failed to create database dump"
    echo "Please check your database credentials and connection"
    exit 1
fi

