#!/bin/bash

# Quick fix script for server JWT token errors
# Run this on your server: bash quick_fix.sh

set -e  # Exit on error

echo "🔧 Starting server error fixes..."
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Navigate to project directory
cd /home/hrbackendtest/public_html || {
    echo -e "${RED}❌ Failed to navigate to project directory${NC}"
    exit 1
}

echo -e "${GREEN}✅ In project directory${NC}"

# Step 1: Check .env file
echo ""
echo "📋 Step 1: Checking environment variables..."

if [ ! -f .env ]; then
    echo -e "${RED}❌ .env file not found!${NC}"
    exit 1
fi

# Backup .env
cp .env .env.backup.$(date +%Y%m%d_%H%M%S)
echo -e "${GREEN}✅ Backed up .env file${NC}"

# Check for JWT_REFRESH_SECRET
if ! grep -q "^JWT_REFRESH_SECRET=" .env; then
    echo -e "${YELLOW}⚠️  JWT_REFRESH_SECRET not found in .env${NC}"
    echo ""
    echo "Please add this line to your .env file:"
    echo ""
    RANDOM_SECRET=$(openssl rand -base64 64 | tr -d '\n')
    echo "JWT_REFRESH_SECRET=$RANDOM_SECRET"
    echo ""
    echo "Run this command to add it automatically:"
    echo "echo 'JWT_REFRESH_SECRET=$RANDOM_SECRET' >> .env"
    echo ""
    read -p "Press Enter to continue after adding the secret..."
else
    echo -e "${GREEN}✅ JWT_REFRESH_SECRET found in .env${NC}"
fi

# Check for JWT_REFRESH_EXPIRES_IN
if ! grep -q "^JWT_REFRESH_EXPIRES_IN=" .env; then
    echo "JWT_REFRESH_EXPIRES_IN=30d" >> .env
    echo -e "${GREEN}✅ Added JWT_REFRESH_EXPIRES_IN to .env${NC}"
fi

# Step 2: Load database credentials from .env
echo ""
echo "📊 Step 2: Setting up database..."

# Source .env file
export $(grep -v '^#' .env | xargs)

# Create refresh_tokens table
echo "Creating refresh_tokens table..."

mysql -u "$DB_USER" -p"$DB_PASSWORD" "$DB_NAME" << 'EOF'
CREATE TABLE IF NOT EXISTS refresh_tokens (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  token TEXT NOT NULL,
  expires_at DATETIME NOT NULL,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  INDEX idx_user_id (user_id),
  INDEX idx_expires_at (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
EOF

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✅ Database table created/verified${NC}"
else
    echo -e "${RED}❌ Failed to create database table${NC}"
    echo "Please check your database credentials in .env"
    exit 1
fi

# Step 3: Clean up expired tokens
echo ""
echo "🧹 Step 3: Cleaning up expired tokens..."

mysql -u "$DB_USER" -p"$DB_PASSWORD" "$DB_NAME" -e "DELETE FROM refresh_tokens WHERE expires_at < NOW();" 2>/dev/null || true

echo -e "${GREEN}✅ Cleanup complete${NC}"

# Step 4: Restart PM2
echo ""
echo "🔄 Step 4: Restarting PM2 processes..."

pm2 restart server

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✅ PM2 restarted successfully${NC}"
else
    echo -e "${RED}❌ Failed to restart PM2${NC}"
    exit 1
fi

# Wait for server to start
echo ""
echo "⏳ Waiting for server to start..."
sleep 3

# Step 5: Check logs
echo ""
echo "📝 Step 5: Checking server logs..."
echo ""

pm2 logs server --lines 20 --nostream

echo ""
echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${GREEN}✅ Fix complete!${NC}"
echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo ""
echo "Next steps:"
echo "1. Monitor logs: pm2 logs server"
echo "2. Check for errors: pm2 logs server --err"
echo "3. Test your API endpoints"
echo ""
echo "If you still see JWT errors, verify:"
echo "- JWT_REFRESH_SECRET is set in .env"
echo "- refresh_tokens table exists in database"
echo "- Database connection is working"
