const mysql = require('mysql2/promise');
require('dotenv').config();

const SQL_MIGRATION = `
-- Add is_automatic_clockout field to attendance table
-- This field tracks whether the clockout was automatic (after 12 hours) or manual

ALTER TABLE attendance 
ADD COLUMN is_automatic_clockout BOOLEAN DEFAULT FALSE 
AFTER clock_out_time;
`;

async function runMigration() {
  let connection;
  
  try {
    console.log('🔄 Connecting to MySQL server...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      port: process.env.DB_PORT || 3306,
      database: process.env.DB_NAME || 'faceml_db',
      multipleStatements: true
    });

    console.log('✓ Connected to MySQL server');
    console.log('🔄 Running migration: Add is_automatic_clockout field...');

    // Execute migration - add column
    await connection.query(SQL_MIGRATION);
    console.log('✅ Added is_automatic_clockout field to attendance table');
    
    // Add index separately (check if it exists first)
    try {
      await connection.query(`
        CREATE INDEX idx_automatic_clockout ON attendance(is_automatic_clockout)
      `);
      console.log('✅ Created index idx_automatic_clockout');
    } catch (indexError) {
      if (indexError.code === 'ER_DUP_KEYNAME') {
        console.log('⚠️  Index idx_automatic_clockout already exists (safe to ignore)');
      } else {
        throw indexError;
      }
    }
    
    console.log('✅ Migration completed successfully!');

  } catch (error) {
    if (error.code === 'ER_DUP_FIELDNAME') {
      console.log('⚠️  Field is_automatic_clockout already exists. Migration may have already been run.');
      console.log('   This is safe to ignore if the field was added previously.');
    } else if (error.code === 'ER_DUP_KEYNAME') {
      console.log('⚠️  Index idx_automatic_clockout already exists. Migration may have already been run.');
      console.log('   This is safe to ignore if the index was created previously.');
    } else {
      console.error('❌ Migration failed:', error.message);
      console.error('   Error code:', error.code);
      process.exit(1);
    }
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

runMigration();

