const mysql = require('mysql2/promise');
const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '..', '.env') });

/**
 * Alters the staff table to support encrypted data
 * Changes VARCHAR columns to TEXT to accommodate encrypted data
 */
async function alterStaffTable() {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
    });

    console.log('✓ Connected to database');
    console.log('🔄 Altering staff table for encrypted data...');

    // Alter columns to TEXT type to store encrypted data
    const alterQueries = [
      `ALTER TABLE staff MODIFY COLUMN full_name TEXT COMMENT 'Encrypted'`,
      `ALTER TABLE staff MODIFY COLUMN email TEXT COMMENT 'Encrypted'`,
      `ALTER TABLE staff MODIFY COLUMN phone TEXT COMMENT 'Encrypted'`,
      `ALTER TABLE staff MODIFY COLUMN face_embeddings LONGTEXT COMMENT 'Encrypted string (was JSON)'`,
    ];

    for (const query of alterQueries) {
      console.log(`  Executing: ${query}`);
      await connection.query(query);
    }

    console.log('✓ Staff table altered successfully');
    console.log('\n✅ Database schema migration completed!\n');
    console.log('The following columns are now ready for encrypted data:');
    console.log('  - full_name (TEXT)');
    console.log('  - email (TEXT)');
    console.log('  - phone (TEXT)');
    console.log('  - face_embeddings (LONGTEXT)\n');

  } catch (error) {
    console.error('❌ Schema migration failed:', error.message);
    console.error('Full error:', error);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('Database connection closed.');
    }
  }
}

// Run the migration
alterStaffTable();
