const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
require('dotenv').config();

/** DUMMY DATA SCRIPT for faceml */
async function seedDummyData() {
  let connection;
  try {
    console.log('🔄 Connecting to MySQL...');
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306
    });
    console.log('✓ Connected!');

    // 0. DELETE PREVIOUS DEMO DATA (except shop/admin)
    await connection.query('DELETE FROM attendance');
    await connection.query('DELETE FROM staff');
    await connection.query('DELETE FROM branches WHERE name != "Main Branch"');
    // Optionally: Don't remove shops/admin unless needed

    // 1. Create shop admin (if not exists)
    const adminEmail = 'shop@gmail.com';
    const adminPass = '123456';
    const adminName = 'Test Demo Shop Admin';
    let [rows] = await connection.query('SELECT id FROM users WHERE email = ?', [adminEmail]);
    let adminId;
    if (rows.length === 0) {
      const hash = await bcrypt.hash(adminPass, 10);
      const [{ insertId }] = await connection.query(
        'INSERT INTO users (email, password, full_name, role) VALUES (?, ?, ?, ?)',
        [adminEmail, hash, adminName, 'shop_admin']
      );
      adminId = insertId;
      console.log('✓ Created shop admin:', adminEmail);
    } else {
      adminId = rows[0].id;
      console.log('✓ Shop admin exists:', adminEmail);
    }

    // 2. Create or get shop for admin
    [rows] = await connection.query('SELECT id FROM shops WHERE admin_id = ?', [adminId]);
    let shopId;
    if (rows.length === 0) {
      const [{ insertId }] = await connection.query(
        'INSERT INTO shops (name, admin_id, address) VALUES (?, ?, ?)',
        ["Demo Shop", adminId, "123 Demo Street"]
      );
      shopId = insertId;
      console.log('✓ Created shop.');
    } else {
      shopId = rows[0].id;
      console.log('✓ Shop exists.');
    }

    // 3. Create/get main branch for shop
    [rows] = await connection.query('SELECT id FROM branches WHERE shop_id = ? AND name = ?', [shopId, 'Main Branch']);
    let branchId;
    if (rows.length === 0) {
      const [{ insertId }] = await connection.query(
        'INSERT INTO branches (shop_id, name, address) VALUES (?, ?, ?)',
        [shopId, "Main Branch", "123 Demo Street"]
      );
      branchId = insertId;
      console.log('✓ Created main branch.');
    } else {
      branchId = rows[0].id;
      console.log('✓ Main branch exists.');
    }

    // 4. Create 5 staff accounts
    const staffList = [
      { code: 'STF001', name: 'Alice - 10 Days 10-5', hourlyRate: 10.00, email: 'alice@shop.com', phone: '+110000001' },
      { code: 'STF002', name: 'Bob - 45 Days 9-5', hourlyRate: 15.00, email: 'bob@shop.com', phone: '+110000002' },
      { code: 'STF003', name: 'Charlie - Variable Hours', hourlyRate: 15.00, email: 'charlie@shop.com', phone: '+110000003' },
      { code: 'STF004', name: 'Derek - Some Leaves', hourlyRate: 15.00, email: 'derek@shop.com', phone: '+110000004' },
      { code: 'STF005', name: 'Eva - 15 Days 10-5', hourlyRate: 10.00, email: 'eva@shop.com', phone: '+110000005' },
    ];
    const staffIds = {};
    for (const s of staffList) {
      [rows] = await connection.query('SELECT id FROM staff WHERE staff_code = ?', [s.code]);
      if (rows.length === 0) {
        const [{ insertId }] = await connection.query(
          'INSERT INTO staff (branch_id, staff_code, full_name, email, phone, position, hourly_rate) VALUES (?, ?, ?, ?, ?, ?, ?)',
          [branchId, s.code, s.name, s.email, s.phone, `Staff $${s.hourlyRate}/hr`, s.hourlyRate]
        );
        staffIds[s.code] = insertId;
        console.log('✓ Created staff:', s.code, 'phone:', s.phone, 'hourly:', s.hourlyRate);
      } else {
        await connection.query('UPDATE staff SET hourly_rate = ?, position = ?, phone = ? WHERE id = ?', [s.hourlyRate, `Staff $${s.hourlyRate}/hr`, s.phone, rows[0].id]);
        staffIds[s.code] = rows[0].id;
        console.log('✓ Staff exists (updated):', s.code, 'phone:', s.phone, 'hourly:', s.hourlyRate);
      }
    }

    // 5. Create realistic attendance
    const attendanceCheck = async (staffId, dateStr) => {
      const [rowA] = await connection.query('SELECT id FROM attendance WHERE staff_id = ? AND attendance_date = ?', [staffId, dateStr]);
      return rowA.length > 0;
    };
    const insertAttendance = async (staffId, inTime, outTime, date, notes = '') => {
      const already = await attendanceCheck(staffId, date);
      if (already) return;
      const total = ((new Date(outTime) - new Date(inTime)) / 3600000).toFixed(2);
      await connection.query(
        'INSERT INTO attendance (staff_id, branch_id, clock_in_time, clock_out_time, total_hours, attendance_date, status, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
        [staffId, branchId, inTime, outTime, total, date, 'present', notes]
      );
    };

    const now = new Date();

    // STAFF 1: 10 working days, 10:00 to 17:00 (5pm), $10/hr
    console.log('⏳ Adding attendance for Staff 1 (10 days, 10-5)...');
    let count1 = 0;
    for (let j = 0; j < 60 && count1 < 10; j++) {
      let day = new Date(now.getTime() - j * 86400000);
      if (day.getDay() === 0 || day.getDay() === 6) continue; // skip weekends
      const ymd = day.toISOString().split('T')[0];
      await insertAttendance(staffIds['STF001'], ymd+' 10:00:00', ymd+' 17:00:00', ymd, 'Regular shift 10-5');
      count1++;
    }

    // STAFF 2: 45 working days, 9:00 to 17:00 (5pm), $15/hr
    console.log('⏳ Adding attendance for Staff 2 (45 days, 9-5)...');
    let count2 = 0;
    for (let j = 0; j < 90 && count2 < 45; j++) {
      let day = new Date(now.getTime() - j * 86400000);
      if (day.getDay() === 0 || day.getDay() === 6) continue; // skip weekends
      const ymd = day.toISOString().split('T')[0];
      await insertAttendance(staffIds['STF002'], ymd+' 09:00:00', ymd+' 17:00:00', ymd, 'Regular shift 9-5');
      count2++;
    }

    // STAFF 3: 30 working days, $15/hr, variable times (9-5, 8-4, 9-3)
    console.log('⏳ Adding attendance for Staff 3 (30 days, variable hours)...');
    const timeVariations = [
      { in: '09:00:00', out: '17:00:00', note: 'Shift 9-5' },
      { in: '08:00:00', out: '16:00:00', note: 'Shift 8-4' },
      { in: '09:00:00', out: '15:00:00', note: 'Shift 9-3' }
    ];
    let count3 = 0;
    for (let j = 0; j < 60 && count3 < 30; j++) {
      let day = new Date(now.getTime() - j * 86400000);
      if (day.getDay() === 0 || day.getDay() === 6) continue; // skip weekends
      const ymd = day.toISOString().split('T')[0];
      const variation = timeVariations[count3 % 3]; // rotate through time variations
      await insertAttendance(staffIds['STF003'], ymd+' '+variation.in, ymd+' '+variation.out, ymd, variation.note);
      count3++;
    }

    // STAFF 4: 45 working days, 9:00 to 17:00 (5pm), $15/hr, but skip some Mondays and Thursdays (leaves)
    console.log('⏳ Adding attendance for Staff 4 (45 days with some Mon/Thu leaves)...');
    let count4 = 0;
    let daysChecked = 0;
    for (let j = 0; j < 120 && count4 < 45; j++) {
      let day = new Date(now.getTime() - j * 86400000);
      if (day.getDay() === 0 || day.getDay() === 6) continue; // skip weekends
      daysChecked++;
      const dow = day.getDay();
      // Skip some Mondays (1) and Thursdays (4) - every 3rd occurrence
      if ((dow === 1 || dow === 4) && daysChecked % 3 === 0) {
        console.log(`  → Skipping leave on ${day.toISOString().split('T')[0]} (${dow === 1 ? 'Monday' : 'Thursday'})`);
        continue;
      }
      const ymd = day.toISOString().split('T')[0];
      await insertAttendance(staffIds['STF004'], ymd+' 09:00:00', ymd+' 17:00:00', ymd, 'Regular shift 9-5');
      count4++;
    }

    // STAFF 5: 15 working days, 10:00 to 17:00 (5pm), $10/hr
    console.log('⏳ Adding attendance for Staff 5 (15 days, 10-5)...');
    let count5 = 0;
    for (let j = 0; j < 60 && count5 < 15; j++) {
      let day = new Date(now.getTime() - j * 86400000);
      if (day.getDay() === 0 || day.getDay() === 6) continue; // skip weekends
      const ymd = day.toISOString().split('T')[0];
      await insertAttendance(staffIds['STF005'], ymd+' 10:00:00', ymd+' 17:00:00', ymd, 'Regular shift 10-5');
      count5++;
    }

    console.log('✓ All dummy data seeded!');
    console.log(`  Staff 1: ${count1} days (10-5, $10/hr)`);
    console.log(`  Staff 2: ${count2} days (9-5, $15/hr)`);
    console.log(`  Staff 3: ${count3} days (variable hours, $15/hr)`);
    console.log(`  Staff 4: ${count4} days (9-5 with leaves, $15/hr)`);
    console.log(`  Staff 5: ${count5} days (10-5, $10/hr)`);
  } catch (err) {
    console.error('❌ Dummy data error:', err.message);
  } finally {
    if (connection) await connection.end();
  }
}

seedDummyData();

