const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

/**
 * Database Dump Generator for FaceML
 * 
 * This script generates SQL dump files that can be imported into:
 * - cPanel/Shared hosting (via phpMyAdmin)
 * - VPS/Dedicated servers (via mysql command line)
 * 
 * Usage:
 *   node scripts/dumpDatabase.js [output_type]
 * 
 * Options:
 *   cpanel  - Generate dump for cPanel (no CREATE DATABASE)
 *   full    - Generate full dump with CREATE DATABASE (default)
 * 
 * Examples:
 *   node scripts/dumpDatabase.js cpanel
 *   node scripts/dumpDatabase.js full
 */

async function getTableStructure(connection, tableName) {
  const [columns] = await connection.query(`
    SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT, 
           COLUMN_TYPE, EXTRA, COLUMN_KEY, COLUMN_COMMENT
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?
    ORDER BY ORDINAL_POSITION
  `, [process.env.DB_NAME || 'faceml_db', tableName]);

  return columns;
}

async function getTableIndexes(connection, tableName) {
  const [indexes] = await connection.query(`
    SELECT INDEX_NAME, COLUMN_NAME, NON_UNIQUE, SEQ_IN_INDEX
    FROM INFORMATION_SCHEMA.STATISTICS
    WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?
    ORDER BY INDEX_NAME, SEQ_IN_INDEX
  `, [process.env.DB_NAME || 'faceml_db', tableName]);

  return indexes;
}

async function getForeignKeys(connection, tableName) {
  const [fks] = await connection.query(`
    SELECT CONSTRAINT_NAME, COLUMN_NAME, REFERENCED_TABLE_NAME, 
           REFERENCED_COLUMN_NAME
    FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
    WHERE TABLE_SCHEMA = ? 
      AND TABLE_NAME = ?
      AND REFERENCED_TABLE_NAME IS NOT NULL
    ORDER BY CONSTRAINT_NAME
  `, [process.env.DB_NAME || 'faceml_db', tableName]);

  return fks;
}

function generateColumnDefinition(col) {
  let def = `\`${col.COLUMN_NAME}\``;
  
  // Handle column type
  if (col.DATA_TYPE === 'enum' || col.DATA_TYPE === 'set') {
    def += ` ${col.COLUMN_TYPE}`;
  } else {
    def += ` ${col.DATA_TYPE.toUpperCase()}`;
    // Add length/precision if it's in COLUMN_TYPE
    const typeMatch = col.COLUMN_TYPE.match(/\((\d+(?:,\d+)?)\)/);
    if (typeMatch) {
      def += `(${typeMatch[1]})`;
    }
  }
  
  if (col.IS_NULLABLE === 'NO') {
    def += ' NOT NULL';
  }
  
  if (col.EXTRA === 'auto_increment') {
    def += ' AUTO_INCREMENT';
  }
  
  if (col.COLUMN_DEFAULT !== null) {
    if (col.COLUMN_DEFAULT === 'CURRENT_TIMESTAMP') {
      def += ` DEFAULT CURRENT_TIMESTAMP`;
    } else if (col.DATA_TYPE === 'timestamp' && col.COLUMN_DEFAULT === 'CURRENT_TIMESTAMP') {
      def += ` DEFAULT CURRENT_TIMESTAMP`;
    } else {
      def += ` DEFAULT ${typeof col.COLUMN_DEFAULT === 'string' ? `'${col.COLUMN_DEFAULT}'` : col.COLUMN_DEFAULT}`;
    }
  } else if (col.IS_NULLABLE === 'YES' && col.DATA_TYPE !== 'timestamp') {
    def += ' DEFAULT NULL';
  }
  
  if (col.COLUMN_KEY === 'PRI') {
    def += ' PRIMARY KEY';
  } else if (col.COLUMN_KEY === 'UNI') {
    def += ' UNIQUE';
  }
  
  return def;
}

function generateCreateTableSQL(connection, tableName, columns, indexes, foreignKeys) {
  let sql = `-- ============================================\n`;
  sql += `-- Table: ${tableName}\n`;
  sql += `-- ============================================\n`;
  sql += `DROP TABLE IF EXISTS \`${tableName}\`;\n`;
  sql += `CREATE TABLE \`${tableName}\` (\n`;
  
  // Column definitions
  const columnDefs = columns.map(col => {
    const def = generateColumnDefinition(col);
    return `  ${def}`;
  });
  
  sql += columnDefs.join(',\n');
  
  // Add indexes (excluding primary key and unique constraints already in column)
  const indexMap = new Map();
  indexes.forEach(idx => {
    if (idx.INDEX_NAME === 'PRIMARY') return;
    if (!indexMap.has(idx.INDEX_NAME)) {
      indexMap.set(idx.INDEX_NAME, []);
    }
    indexMap.get(idx.INDEX_NAME).push(idx.COLUMN_NAME);
  });
  
  // Add unique keys
  indexMap.forEach((cols, idxName) => {
    const isUnique = indexes.find(i => i.INDEX_NAME === idxName)?.NON_UNIQUE === 0;
    if (isUnique && !columns.find(c => c.COLUMN_KEY === 'UNI' && cols.includes(c.COLUMN_NAME))) {
      sql += `,\n  UNIQUE KEY \`${idxName}\` (\`${cols.join('`, `')}\`)`;
    } else if (!isUnique) {
      sql += `,\n  KEY \`${idxName}\` (\`${cols.join('`, `')}\`)`;
    }
  });
  
  // Add foreign keys
  foreignKeys.forEach(fk => {
    sql += `,\n  CONSTRAINT \`${fk.CONSTRAINT_NAME}\` FOREIGN KEY (\`${fk.COLUMN_NAME}\`) REFERENCES \`${fk.REFERENCED_TABLE_NAME}\` (\`${fk.REFERENCED_COLUMN_NAME}\`) ON DELETE CASCADE`;
  });
  
  sql += `\n) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;\n\n`;
  
  return sql;
}

async function generateDump(outputType = 'full') {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306
    });

    console.log('✓ Connected to database');
    
    // Get all tables
    const [tables] = await connection.query(`
      SELECT TABLE_NAME
      FROM INFORMATION_SCHEMA.TABLES
      WHERE TABLE_SCHEMA = ?
      ORDER BY TABLE_NAME
    `, [process.env.DB_NAME || 'faceml_db']);

    console.log(`✓ Found ${tables.length} tables`);
    
    // Get all foreign key relationships to sort tables by dependencies
    const [fkRelations] = await connection.query(`
      SELECT 
        TABLE_NAME,
        REFERENCED_TABLE_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
      WHERE TABLE_SCHEMA = ?
        AND REFERENCED_TABLE_NAME IS NOT NULL
    `, [process.env.DB_NAME || 'faceml_db']);
    
    // Create dependency map
    const dependencies = new Map();
    const allTableNames = tables.map(t => t.TABLE_NAME);
    allTableNames.forEach(tableName => {
      dependencies.set(tableName, []);
    });
    
    fkRelations.forEach(fk => {
      if (dependencies.has(fk.TABLE_NAME) && fk.REFERENCED_TABLE_NAME) {
        dependencies.get(fk.TABLE_NAME).push(fk.REFERENCED_TABLE_NAME);
      }
    });
    
    // Topological sort to order tables by dependencies
    const sortedTables = [];
    const visited = new Set();
    const visiting = new Set();
    
    function visit(tableName) {
      if (visiting.has(tableName)) {
        // Circular dependency - shouldn't happen but handle gracefully
        return;
      }
      if (visited.has(tableName)) {
        return;
      }
      visiting.add(tableName);
      const deps = dependencies.get(tableName) || [];
      deps.forEach(dep => {
        if (allTableNames.includes(dep)) {
          visit(dep);
        }
      });
      visiting.delete(tableName);
      visited.add(tableName);
      sortedTables.push(tableName);
    }
    
    allTableNames.forEach(tableName => {
      if (!visited.has(tableName)) {
        visit(tableName);
      }
    });
    
    // Filter tables to only include those that exist and sort them
    const orderedTables = tables.filter(t => sortedTables.includes(t.TABLE_NAME));
    orderedTables.sort((a, b) => {
      return sortedTables.indexOf(a.TABLE_NAME) - sortedTables.indexOf(b.TABLE_NAME);
    });
    
    console.log(`✓ Ordered ${orderedTables.length} tables by dependencies`);
    
    // Build SQL dump
    let sqlDump = '';
    
    // Header
    const timestamp = new Date().toISOString();
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- FaceML Database Export\n`;
    sqlDump += `-- Generated: ${timestamp}\n`;
    sqlDump += `-- Output Type: ${outputType === 'cpanel' ? 'cPanel/Shared Hosting' : 'Full (VPS/Dedicated)'}\n`;
    sqlDump += `-- ============================================\n\n`;
    
    if (outputType === 'cpanel') {
      sqlDump += `-- IMPORTANT: Before importing this file:\n`;
      sqlDump += `-- 1. Create database through cPanel MySQL Databases\n`;
      sqlDump += `-- 2. Note the full database name (usually: cpses_xxxxx_faceml or similar)\n`;
      sqlDump += `-- 3. Select that database in phpMyAdmin before importing\n`;
      sqlDump += `-- 4. Then import this file\n`;
      sqlDump += `-- ============================================\n\n`;
      sqlDump += `-- No CREATE DATABASE command - you must create it via cPanel first\n`;
      sqlDump += `-- Then select it in phpMyAdmin and import this file\n\n`;
    } else {
      sqlDump += `-- Create database\n`;
      sqlDump += `CREATE DATABASE IF NOT EXISTS \`${process.env.DB_NAME || 'faceml_db'}\` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\n`;
      sqlDump += `USE \`${process.env.DB_NAME || 'faceml_db'}\`;\n\n`;
    }
    
    // Process each table in dependency order
    for (const table of orderedTables) {
      const tableName = table.TABLE_NAME;
      console.log(`  Processing table: ${tableName}`);
      
      const columns = await getTableStructure(connection, tableName);
      const indexes = await getTableIndexes(connection, tableName);
      const foreignKeys = await getForeignKeys(connection, tableName);
      
      sqlDump += generateCreateTableSQL(connection, tableName, columns, indexes, foreignKeys);
    }
    
    // Add default super admin
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Insert Default Super Admin\n`;
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Note: This is a placeholder password hash\n`;
    sqlDump += `-- After importing, you MUST run: node scripts/initDatabase.js\n`;
    sqlDump += `-- to create the proper password hash\n`;
    sqlDump += `INSERT INTO \`users\` (\`email\`, \`password\`, \`full_name\`, \`role\`, \`is_active\`) VALUES\n`;
    sqlDump += `('admin@faceml.com', '$2a$10$YourHashedPasswordHere', 'Super Admin', 'superadmin', true)\n`;
    sqlDump += `ON DUPLICATE KEY UPDATE \`updated_at\` = CURRENT_TIMESTAMP;\n\n`;
    
    // Footer
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Import Complete!\n`;
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Next steps:\n`;
    sqlDump += `-- 1. Update .env file with your database credentials\n`;
    sqlDump += `-- 2. Run: node scripts/initDatabase.js\n`;
    sqlDump += `-- 3. This will create proper super admin password hash\n`;
    sqlDump += `-- 4. Start your server: node server.js\n`;
    sqlDump += `-- ============================================\n`;
    
    // Write to file
    const filename = outputType === 'cpanel' 
      ? 'faceml_database_cpanel_LATEST.sql'
      : 'faceml_database_LATEST.sql';
    const filepath = path.join(__dirname, '..', filename);
    
    fs.writeFileSync(filepath, sqlDump, 'utf8');
    
    console.log('\n✅ Database dump generated successfully!');
    console.log(`📁 File: ${filepath}`);
    console.log(`📊 Tables exported: ${orderedTables.length}`);
    
  } catch (error) {
    console.error('❌ Error generating dump:', error.message);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Get output type from command line
const outputType = process.argv[2] === 'cpanel' ? 'cpanel' : 'full';

generateDump(outputType);

