const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

/**
 * Database Dump Generator with Data for FaceML
 * 
 * This script generates SQL dump files with BOTH structure and data
 * that can be imported into:
 * - cPanel/Shared hosting (via phpMyAdmin)
 * - VPS/Dedicated servers (via mysql command line)
 * 
 * Usage:
 *   node scripts/dumpDatabaseWithData.js [output_type]
 * 
 * Options:
 *   cpanel  - Generate dump for cPanel (no CREATE DATABASE)
 *   full    - Generate full dump with CREATE DATABASE (default)
 * 
 * Examples:
 *   node scripts/dumpDatabaseWithData.js cpanel
 *   node scripts/dumpDatabaseWithData.js full
 */

function escapeValue(value) {
  if (value === null || value === undefined) {
    return 'NULL';
  }
  if (typeof value === 'number') {
    return value.toString();
  }
  if (typeof value === 'boolean') {
    return value ? '1' : '0';
  }
  if (value instanceof Date) {
    return `'${value.toISOString().slice(0, 19).replace('T', ' ')}'`;
  }
  // Escape string values
  const str = String(value);
  return `'${str.replace(/\\/g, '\\\\').replace(/'/g, "''").replace(/\n/g, '\\n').replace(/\r/g, '\\r')}'`;
}

async function generateDump(outputType = 'full') {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
      multipleStatements: true
    });

    console.log('✓ Connected to database');
    
    // Get all tables
    const [tables] = await connection.query(`
      SELECT TABLE_NAME
      FROM INFORMATION_SCHEMA.TABLES
      WHERE TABLE_SCHEMA = ?
        AND TABLE_TYPE = 'BASE TABLE'
      ORDER BY TABLE_NAME
    `, [process.env.DB_NAME || 'faceml_db']);

    console.log(`✓ Found ${tables.length} tables`);
    
    // Get all foreign key relationships to sort tables by dependencies
    const [fkRelations] = await connection.query(`
      SELECT 
        TABLE_NAME,
        REFERENCED_TABLE_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
      WHERE TABLE_SCHEMA = ?
        AND REFERENCED_TABLE_NAME IS NOT NULL
    `, [process.env.DB_NAME || 'faceml_db']);
    
    // Create dependency map
    const dependencies = new Map();
    const allTableNames = tables.map(t => t.TABLE_NAME);
    allTableNames.forEach(tableName => {
      dependencies.set(tableName, []);
    });
    
    fkRelations.forEach(fk => {
      if (dependencies.has(fk.TABLE_NAME) && fk.REFERENCED_TABLE_NAME) {
        dependencies.get(fk.TABLE_NAME).push(fk.REFERENCED_TABLE_NAME);
      }
    });
    
    // Topological sort to order tables by dependencies
    const sortedTables = [];
    const visited = new Set();
    const visiting = new Set();
    
    function visit(tableName) {
      if (visiting.has(tableName)) {
        return;
      }
      if (visited.has(tableName)) {
        return;
      }
      visiting.add(tableName);
      const deps = dependencies.get(tableName) || [];
      deps.forEach(dep => {
        if (allTableNames.includes(dep)) {
          visit(dep);
        }
      });
      visiting.delete(tableName);
      visited.add(tableName);
      sortedTables.push(tableName);
    }
    
    allTableNames.forEach(tableName => {
      if (!visited.has(tableName)) {
        visit(tableName);
      }
    });
    
    // Filter and sort tables
    const orderedTables = tables.filter(t => sortedTables.includes(t.TABLE_NAME));
    orderedTables.sort((a, b) => {
      return sortedTables.indexOf(a.TABLE_NAME) - sortedTables.indexOf(b.TABLE_NAME);
    });
    
    console.log(`✓ Ordered ${orderedTables.length} tables by dependencies`);
    
    // Build SQL dump
    let sqlDump = '';
    
    // Header
    const timestamp = new Date().toISOString();
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- FaceML Database Export (Structure + Data)\n`;
    sqlDump += `-- Generated: ${timestamp}\n`;
    sqlDump += `-- Output Type: ${outputType === 'cpanel' ? 'cPanel/Shared Hosting' : 'Full (VPS/Dedicated)'}\n`;
    sqlDump += `-- ============================================\n\n`;
    
    // Set SQL mode and charset
    sqlDump += `SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";\n`;
    sqlDump += `SET time_zone = "+00:00";\n\n`;
    
    if (outputType === 'cpanel') {
      sqlDump += `-- IMPORTANT: Before importing this file:\n`;
      sqlDump += `-- 1. Create database through cPanel MySQL Databases\n`;
      sqlDump += `-- 2. Note the full database name (usually: cpses_xxxxx_faceml or similar)\n`;
      sqlDump += `-- 3. Select that database in phpMyAdmin before importing\n`;
      sqlDump += `-- 4. Then import this file\n`;
      sqlDump += `-- ============================================\n\n`;
    } else {
      sqlDump += `-- Create database\n`;
      sqlDump += `CREATE DATABASE IF NOT EXISTS \`${process.env.DB_NAME || 'faceml_db'}\` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\n`;
      sqlDump += `USE \`${process.env.DB_NAME || 'faceml_db'}\`;\n\n`;
    }
    
    // Disable foreign key checks temporarily
    sqlDump += `-- Disable foreign key checks for import\n`;
    sqlDump += `SET FOREIGN_KEY_CHECKS=0;\n\n`;
    
    // Process each table
    for (const table of orderedTables) {
      const tableName = table.TABLE_NAME;
      console.log(`  Processing table: ${tableName}`);
      
      // Get table structure
      const [createTable] = await connection.query(`SHOW CREATE TABLE \`${tableName}\``);
      const createTableSQL = createTable[0]['Create Table'];
      
      sqlDump += `-- ============================================\n`;
      sqlDump += `-- Table structure for table: ${tableName}\n`;
      sqlDump += `-- ============================================\n`;
      sqlDump += `DROP TABLE IF EXISTS \`${tableName}\`;\n`;
      sqlDump += `${createTableSQL};\n\n`;
      
      // Get table data
      const [rows] = await connection.query(`SELECT * FROM \`${tableName}\``);
      
      if (rows.length > 0) {
        sqlDump += `-- ============================================\n`;
        sqlDump += `-- Dumping data for table: ${tableName}\n`;
        sqlDump += `-- ============================================\n`;
        
        // Get column names
        const [columns] = await connection.query(`SHOW COLUMNS FROM \`${tableName}\``);
        const columnNames = columns.map(col => `\`${col.Field}\``);
        
        // Generate INSERT statements
        sqlDump += `INSERT INTO \`${tableName}\` (${columnNames.join(', ')}) VALUES\n`;
        
        const values = rows.map((row, index) => {
          const rowValues = columnNames.map(colName => {
            const cleanName = colName.replace(/`/g, '');
            return escapeValue(row[cleanName]);
          });
          const comma = index < rows.length - 1 ? ',' : ';';
          return `(${rowValues.join(', ')})${comma}`;
        });
        
        sqlDump += values.join('\n');
        sqlDump += `\n\n`;
        
        console.log(`    ✓ Exported ${rows.length} rows`);
      } else {
        sqlDump += `-- Table ${tableName} is empty\n\n`;
        console.log(`    ✓ Table is empty`);
      }
    }
    
    // Re-enable foreign key checks
    sqlDump += `-- Re-enable foreign key checks\n`;
    sqlDump += `SET FOREIGN_KEY_CHECKS=1;\n\n`;
    
    // Footer
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Import Complete!\n`;
    sqlDump += `-- ============================================\n`;
    sqlDump += `-- Next steps:\n`;
    sqlDump += `-- 1. Update .env file with your database credentials\n`;
    sqlDump += `-- 2. Run: node scripts/initDatabase.js (if needed)\n`;
    sqlDump += `-- 3. Start your server: node server.js\n`;
    sqlDump += `-- ============================================\n`;
    
    // Write to file
    const fileTimestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const filename = outputType === 'cpanel' 
      ? `faceml_database_cpanel_${fileTimestamp}.sql`
      : `faceml_database_full_${fileTimestamp}.sql`;
    const filepath = path.join(__dirname, '..', filename);
    
    fs.writeFileSync(filepath, sqlDump, 'utf8');
    
    const stats = fs.statSync(filepath);
    const fileSizeMB = (stats.size / (1024 * 1024)).toFixed(2);
    
    console.log('\n✅ Database dump generated successfully!');
    console.log(`📁 File: ${filepath}`);
    console.log(`📊 Tables exported: ${orderedTables.length}`);
    console.log(`💾 File size: ${fileSizeMB} MB`);
    
  } catch (error) {
    console.error('❌ Error generating dump:', error.message);
    console.error(error.stack);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Get output type from command line
const outputType = process.argv[2] === 'cpanel' ? 'cpanel' : 'full';

generateDump(outputType);

