const mysql = require('mysql2/promise');
const fs = require('fs').promises;
const path = require('path');
const encryptionService = require('../services/encryptionService');
require('dotenv').config({ path: path.join(__dirname, '..', '.env') });

/**
 * Encrypts all existing staff data in the database
 * Creates a backup before encryption
 */
async function encryptExistingData() {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
    });

    console.log('✓ Connected to database');
    
    // Fetch all staff records
    console.log('🔄 Fetching existing staff records...');
    const [staffRecords] = await connection.query(
      'SELECT id, full_name, email, phone, face_embeddings FROM staff'
    );
    
    if (staffRecords.length === 0) {
      console.log('ℹ️  No staff records found to encrypt.');
      return;
    }
    
    console.log(`✓ Found ${staffRecords.length} staff record(s)`);
    
    // Create backup
    console.log('🔄 Creating backup...');
    const backupDir = path.join(__dirname, '..', 'backups');
    
    try {
      await fs.mkdir(backupDir, { recursive: true });
    } catch (err) {
      // Directory might already exist
    }
    
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const backupFile = path.join(backupDir, `staff_backup_${timestamp}.json`);
    
    await fs.writeFile(
      backupFile,
      JSON.stringify(staffRecords, null, 2),
      'utf8'
    );
    
    console.log(`✓ Backup created: ${backupFile}`);
    
    // Encrypt each record
    console.log('🔄 Encrypting staff records...');
    let successCount = 0;
    let errorCount = 0;
    
    for (const staff of staffRecords) {
      try {
        // Check if data is already encrypted (contains ':' which is our delimiter)
        const isAlreadyEncrypted = 
          (staff.full_name && staff.full_name.includes(':')) ||
          (staff.email && staff.email.includes(':')) ||
          (staff.phone && staff.phone.includes(':'));
        
        if (isAlreadyEncrypted) {
          console.log(`  ⚠️  Staff ID ${staff.id} appears to be already encrypted, skipping...`);
          continue;
        }
        
        // Encrypt sensitive fields
        const encryptedName = staff.full_name 
          ? encryptionService.encrypt(staff.full_name) 
          : null;
        
        const encryptedEmail = staff.email 
          ? encryptionService.encrypt(staff.email) 
          : null;
        
        const encryptedPhone = staff.phone 
          ? encryptionService.encrypt(staff.phone) 
          : null;
        
        // Handle face embeddings
        let encryptedEmbeddings = null;
        if (staff.face_embeddings) {
          try {
            // face_embeddings is already JSON, convert to string and encrypt
            const embeddingsStr = typeof staff.face_embeddings === 'string'
              ? staff.face_embeddings
              : JSON.stringify(staff.face_embeddings);
            
            encryptedEmbeddings = encryptionService.encrypt(embeddingsStr);
          } catch (err) {
            console.warn(`  ⚠️  Failed to encrypt face embeddings for staff ID ${staff.id}:`, err.message);
          }
        }
        
        // Update database with encrypted data
        await connection.query(
          `UPDATE staff 
           SET full_name = ?, 
               email = ?, 
               phone = ?, 
               face_embeddings = ?
           WHERE id = ?`,
          [
            encryptedName,
            encryptedEmail,
            encryptedPhone,
            encryptedEmbeddings,
            staff.id
          ]
        );
        
        successCount++;
        console.log(`  ✓ Encrypted staff ID ${staff.id}: ${staff.full_name}`);
        
      } catch (error) {
        errorCount++;
        console.error(`  ❌ Failed to encrypt staff ID ${staff.id}:`, error.message);
      }
    }
    
    console.log('\n✅ Encryption completed!');
    console.log(`  Successfully encrypted: ${successCount} record(s)`);
    console.log(`  Errors: ${errorCount} record(s)`);
    console.log(`  Backup location: ${backupFile}\n`);
    
    if (errorCount > 0) {
      console.log('⚠️  Some records failed to encrypt. Please check the errors above.');
      console.log('   The backup file can be used to restore if needed.\n');
    }

  } catch (error) {
    console.error('❌ Encryption failed:', error.message);
    console.error('Full error:', error);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('Database connection closed.');
    }
  }
}

// Run the encryption
encryptExistingData();
