const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
require('dotenv').config();

/**
 * Fix Admin Password Script
 * 
 * This script updates the admin user's password hash in the database.
 * Use this after importing a SQL dump with placeholder password hash.
 * 
 * Usage:
 *   node scripts/fixAdminPassword.js
 * 
 * Environment variables (from .env):
 *   DB_HOST, DB_USER, DB_PASSWORD, DB_NAME, DB_PORT
 *   SUPER_ADMIN_EMAIL (default: admin@faceml.com)
 *   SUPER_ADMIN_PASSWORD (default: Admin@123)
 */

async function fixAdminPassword() {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    console.log(`   Host: ${process.env.DB_HOST || 'localhost'}`);
    console.log(`   Database: ${process.env.DB_NAME || 'faceml_db'}`);
    console.log(`   User: ${process.env.DB_USER || 'root'}`);
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306
    });

    console.log('✓ Connected to database\n');
    
    const adminEmail = process.env.SUPER_ADMIN_EMAIL || 'admin@faceml.com';
    const adminPassword = process.env.SUPER_ADMIN_PASSWORD || 'Admin@123';
    
    // Check if user exists
    const [users] = await connection.query(
      'SELECT id, email, password FROM users WHERE email = ?',
      [adminEmail]
    );

    if (users.length === 0) {
      console.log(`❌ User with email "${adminEmail}" not found.`);
      console.log('   Creating new admin user...');
      
      // Create new admin user
      const hashedPassword = await bcrypt.hash(adminPassword, 10);
      await connection.query(
        'INSERT INTO users (email, password, full_name, role, is_active) VALUES (?, ?, ?, ?, ?)',
        [adminEmail, hashedPassword, 'Super Admin', 'superadmin', true]
      );
      
      console.log('✓ Admin user created successfully!\n');
    } else {
      console.log(`✓ Found user: ${adminEmail}`);
      console.log('🔄 Updating password hash...');
      
      // Generate new password hash
      const hashedPassword = await bcrypt.hash(adminPassword, 10);
      
      // Update password
      await connection.query(
        'UPDATE users SET password = ?, updated_at = CURRENT_TIMESTAMP WHERE email = ?',
        [hashedPassword, adminEmail]
      );
      
      console.log('✓ Password hash updated successfully!\n');
    }

    console.log('✅ Password fix completed!\n');
    console.log('Login Credentials:');
    console.log('  Email:', adminEmail);
    console.log('  Password:', adminPassword);
    console.log('\n⚠️  IMPORTANT: Change this password after first login!\n');

  } catch (error) {
    console.error('❌ Error fixing admin password:', error.message);
    
    if (error.code === 'ER_ACCESS_DENIED_ERROR') {
      console.error('\n💡 Tip: Check your .env file database credentials:');
      console.error('   DB_HOST, DB_USER, DB_PASSWORD, DB_NAME, DB_PORT');
    } else if (error.code === 'ER_BAD_DB_ERROR') {
      console.error('\n💡 Tip: Make sure the database exists and the name in .env is correct.');
    } else if (error.code === 'ECONNREFUSED') {
      console.error('\n💡 Tip: Check that MySQL is running and accessible.');
    }
    
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Run the script
fixAdminPassword();

