const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

async function runMigration() {
  let connection;
  
  try {
    // Create database connection
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
      multipleStatements: true
    });

    console.log('✓ Connected to database');
    console.log(`✓ Database: ${process.env.DB_NAME || 'faceml_db'}`);

    // Read SQL migration file
    const sqlFile = path.join(__dirname, 'fix_all_missing_columns.sql');
    const sql = fs.readFileSync(sqlFile, 'utf8');

    console.log('✓ Reading migration file...');
    console.log('✓ Executing migration...\n');

    // Execute migration
    const results = await connection.query(sql);
    
    console.log('✓ Migration executed successfully!\n');

    // Verify the changes
    console.log('Verifying changes...\n');

    // Check feature_flags.shop_id
    const [featureFlagsColumns] = await connection.query(`
      SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND COLUMN_NAME = 'shop_id'
    `);
    
    if (featureFlagsColumns.length > 0) {
      console.log('✓ feature_flags.shop_id column exists');
    } else {
      console.log('✗ feature_flags.shop_id column is missing');
    }

    // Check feature_flags index
    const [featureFlagsIndex] = await connection.query(`
      SELECT INDEX_NAME 
      FROM INFORMATION_SCHEMA.STATISTICS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'feature_flags' 
      AND INDEX_NAME = 'idx_shop_feature'
    `);
    
    if (featureFlagsIndex.length > 0) {
      console.log('✓ feature_flags.idx_shop_feature index exists');
    } else {
      console.log('✗ feature_flags.idx_shop_feature index is missing');
    }

    // Check refresh_tokens table
    const [refreshTokensTable] = await connection.query(`
      SELECT TABLE_NAME 
      FROM INFORMATION_SCHEMA.TABLES 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'refresh_tokens'
    `);
    
    if (refreshTokensTable.length > 0) {
      console.log('✓ refresh_tokens table exists');
    } else {
      console.log('✗ refresh_tokens table is missing');
    }

    // Check branches.pin
    const [branchesColumns] = await connection.query(`
      SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'branches' 
      AND COLUMN_NAME = 'pin'
    `);
    
    if (branchesColumns.length > 0) {
      console.log('✓ branches.pin column exists');
    } else {
      console.log('✗ branches.pin column is missing');
    }

    console.log('\n✓ All checks completed!');
    console.log('\nPlease restart your PM2 server:');
    console.log('  pm2 restart server\n');

  } catch (error) {
    console.error('✗ Migration failed:', error.message);
    console.error('Error details:', error);
    
    // Don't exit with error if it's a duplicate/already exists error
    const skipErrors = [
      "Duplicate column name",
      "Duplicate key name",
      "Duplicate foreign key",
      "already exists",
      "Duplicate entry"
    ];
    
    const isSkipError = skipErrors.some(err => 
      error.message.includes(err) || 
      error.code === 'ER_DUP_FIELDNAME' || 
      error.code === 'ER_DUP_KEYNAME' ||
      error.code === 'ER_DUP_ENTRY'
    );
    
    if (!isSkipError) {
        console.error('✗ Fatal error - migration failed');
        process.exit(1);
    } else {
        console.log('  (Some elements already exist - this is OK)');
    }
  } finally {
    if (connection) {
      await connection.end();
      console.log('\n✓ Database connection closed');
    }
  }
}

// Run migration
console.log('========================================');
console.log('Database Migration Script');
console.log('Fixing all missing columns and tables');
console.log('========================================\n');

runMigration();

