const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
require('dotenv').config();

/**
 * This script hashes passwords for existing admin accounts
 * Run this after importing the SQL file to properly hash the placeholder passwords
 */

async function hashAdminPasswords() {
  let connection;
  
  try {
    console.log('🔄 Connecting to MySQL database...');
    
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306
    });

    console.log('✓ Connected to database');

    // Hash Super Admin password
    console.log('\n🔄 Hashing Super Admin password...');
    const superAdminEmail = process.env.SUPER_ADMIN_EMAIL || 'admin@faceml.com';
    const superAdminPassword = process.env.SUPER_ADMIN_PASSWORD || 'Admin@123';
    const superAdminHash = await bcrypt.hash(superAdminPassword, 10);

    // Update or insert super admin
    const [superAdminRows] = await connection.query(
      'SELECT id FROM users WHERE email = ?',
      [superAdminEmail]
    );

    if (superAdminRows.length > 0) {
      await connection.query(
        'UPDATE users SET password = ? WHERE email = ?',
        [superAdminHash, superAdminEmail]
      );
      console.log('✓ Super Admin password updated');
    } else {
      await connection.query(
        'INSERT INTO users (email, password, full_name, role) VALUES (?, ?, ?, ?)',
        [superAdminEmail, superAdminHash, 'Super Admin', 'superadmin']
      );
      console.log('✓ Super Admin created');
    }

    // Hash Shop Admin password
    console.log('\n🔄 Hashing Shop Admin password...');
    const shopAdminEmail = process.env.SHOP_ADMIN_EMAIL || 'shop@gmail.com';
    const shopAdminPassword = process.env.SHOP_ADMIN_PASSWORD || '123456';
    const shopAdminHash = await bcrypt.hash(shopAdminPassword, 10);

    // Update or insert shop admin
    const [shopAdminRows] = await connection.query(
      'SELECT id FROM users WHERE email = ?',
      [shopAdminEmail]
    );

    if (shopAdminRows.length > 0) {
      await connection.query(
        'UPDATE users SET password = ? WHERE email = ?',
        [shopAdminHash, shopAdminEmail]
      );
      console.log('✓ Shop Admin password updated');
    } else {
      await connection.query(
        'INSERT INTO users (email, password, full_name, role) VALUES (?, ?, ?, ?)',
        [shopAdminEmail, shopAdminHash, 'Demo Shop Admin', 'shop_admin']
      );
      console.log('✓ Shop Admin created');
    }

    console.log('\n✅ Password hashing completed successfully!\n');
    console.log('═══════════════════════════════════════');
    console.log('📋 Login Credentials:');
    console.log('═══════════════════════════════════════');
    console.log('\n🔑 Super Admin:');
    console.log('   Email:', superAdminEmail);
    console.log('   Password:', superAdminPassword);
    console.log('\n🔑 Shop Admin:');
    console.log('   Email:', shopAdminEmail);
    console.log('   Password:', shopAdminPassword);
    console.log('\n═══════════════════════════════════════');
    console.log('⚠️  IMPORTANT: Change these passwords after first login!');
    console.log('═══════════════════════════════════════\n');

    // Test login
    console.log('🧪 Testing Super Admin login...');
    const [testUser] = await connection.query(
      'SELECT * FROM users WHERE email = ?',
      [superAdminEmail]
    );

    if (testUser.length > 0) {
      const passwordMatch = await bcrypt.compare(superAdminPassword, testUser[0].password);
      if (passwordMatch) {
        console.log('✓ Super Admin password verification successful!');
      } else {
        console.log('❌ Super Admin password verification FAILED!');
      }
    }

    console.log('\n🧪 Testing Shop Admin login...');
    const [testShop] = await connection.query(
      'SELECT * FROM users WHERE email = ?',
      [shopAdminEmail]
    );

    if (testShop.length > 0) {
      const passwordMatch = await bcrypt.compare(shopAdminPassword, testShop[0].password);
      if (passwordMatch) {
        console.log('✓ Shop Admin password verification successful!');
      } else {
        console.log('❌ Shop Admin password verification FAILED!');
      }
    }

    console.log('\n✅ All tests passed! You can now login via API.\n');

  } catch (error) {
    console.error('\n❌ Error:', error.message);
    console.error('\nTroubleshooting:');
    console.error('1. Check your .env file has correct database credentials');
    console.error('2. Verify MySQL server is running');
    console.error('3. Ensure database exists and SQL file was imported');
    console.error('4. Check user has proper permissions\n');
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

hashAdminPasswords();

