const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');
require('dotenv').config();

const SQL_SCHEMA = `
-- Create database if not exists
CREATE DATABASE IF NOT EXISTS ${process.env.DB_NAME || 'faceml_db'};
USE ${process.env.DB_NAME || 'faceml_db'};

-- Users table (for all user types)
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  email VARCHAR(255) UNIQUE NOT NULL,
  password VARCHAR(255) NOT NULL,
  full_name VARCHAR(255) NOT NULL,
  role ENUM('superadmin', 'shop_admin') NOT NULL,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  INDEX idx_email (email),
  INDEX idx_role (role)
);

-- Shops table
CREATE TABLE IF NOT EXISTS shops (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(255) NOT NULL,
  admin_id INT NOT NULL,
  address TEXT,
  contact_phone VARCHAR(20),
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE CASCADE,
  INDEX idx_admin_id (admin_id)
);

-- Branches table
CREATE TABLE IF NOT EXISTS branches (
  id INT AUTO_INCREMENT PRIMARY KEY,
  shop_id INT NOT NULL,
  name VARCHAR(255) NOT NULL,
  address TEXT,
  contact_phone VARCHAR(20),
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (shop_id) REFERENCES shops(id) ON DELETE CASCADE,
  INDEX idx_shop_id (shop_id)
);

-- Staff table
CREATE TABLE IF NOT EXISTS staff (
  id INT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  staff_code VARCHAR(50) UNIQUE NOT NULL,
  full_name VARCHAR(255) NOT NULL,
  email VARCHAR(255),
  phone VARCHAR(20),
  position VARCHAR(100),
  hourly_rate DECIMAL(10,2) DEFAULT 0.00,
  face_embeddings JSON,
  profile_image_path TEXT,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_branch_id (branch_id),
  INDEX idx_staff_code (staff_code)
);

-- Attendance table
CREATE TABLE IF NOT EXISTS attendance (
  id INT AUTO_INCREMENT PRIMARY KEY,
  staff_id INT NOT NULL,
  branch_id INT NOT NULL,
  clock_in_time TIMESTAMP NULL,
  clock_out_time TIMESTAMP NULL,
  clock_in_location VARCHAR(255),
  clock_out_location VARCHAR(255),
  total_hours DECIMAL(5,2),
  attendance_date DATE NOT NULL,
  status ENUM('present', 'absent', 'half_day', 'on_leave') DEFAULT 'present',
  notes TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_staff_id (staff_id),
  INDEX idx_branch_id (branch_id),
  INDEX idx_attendance_date (attendance_date),
  INDEX idx_date_staff (attendance_date, staff_id)
);

-- Work sessions table (for detailed tracking)
CREATE TABLE IF NOT EXISTS work_sessions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  attendance_id INT NOT NULL,
  staff_id INT NOT NULL,
  start_time TIMESTAMP NOT NULL,
  end_time TIMESTAMP NULL,
  duration_minutes INT,
  session_type ENUM('work', 'break', 'overtime') DEFAULT 'work',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (attendance_id) REFERENCES attendance(id) ON DELETE CASCADE,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  INDEX idx_attendance_id (attendance_id),
  INDEX idx_staff_id (staff_id)
);

-- Staff availability table
CREATE TABLE IF NOT EXISTS staff_availability (
  id BIGINT AUTO_INCREMENT PRIMARY KEY,
  staff_id INT NOT NULL,
  branch_id INT NOT NULL,
  availability_date DATE NOT NULL,
  is_available BOOLEAN DEFAULT true,
  start_minutes INT NULL,
  end_minutes INT NULL,
  notes TEXT,
  submitted_by ENUM('staff', 'shop_admin') DEFAULT 'staff',
  submitted_by_id INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY uniq_staff_availability (staff_id, availability_date),
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_branch_date (branch_id, availability_date),
  INDEX idx_staff_date (staff_id, availability_date)
);

-- Roster configuration tables
CREATE TABLE IF NOT EXISTS roster_shift_types (
  id CHAR(36) PRIMARY KEY,
  branch_id INT NOT NULL,
  name VARCHAR(120) NOT NULL,
  start_minutes INT NOT NULL,
  end_minutes INT NOT NULL,
  created_by INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_branch (branch_id)
);

CREATE TABLE IF NOT EXISTS roster_positions (
  id CHAR(36) PRIMARY KEY,
  branch_id INT NOT NULL,
  name VARCHAR(120) NOT NULL,
  created_by INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_branch (branch_id)
);

CREATE TABLE IF NOT EXISTS roster_break_types (
  id CHAR(36) PRIMARY KEY,
  branch_id INT NOT NULL,
  name VARCHAR(120) NOT NULL,
  duration_minutes INT NOT NULL,
  created_by INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  INDEX idx_branch (branch_id)
);

CREATE TABLE IF NOT EXISTS roster_assignments (
  id BIGINT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  shift_date DATE NOT NULL,
  staff_id INT NOT NULL,
  shift_type_id CHAR(36) NOT NULL,
  position_id CHAR(36),
  start_minutes INT NOT NULL,
  end_minutes INT NOT NULL,
  break_type_ids JSON,
  notes TEXT,
  created_by_type ENUM('shop_admin', 'staff') DEFAULT 'shop_admin',
  created_by_id INT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (shift_type_id) REFERENCES roster_shift_types(id) ON DELETE CASCADE,
  FOREIGN KEY (position_id) REFERENCES roster_positions(id) ON DELETE SET NULL,
  INDEX idx_branch_date (branch_id, shift_date),
  INDEX idx_staff_date (staff_id, shift_date),
  INDEX idx_shift_type_date (shift_type_id, shift_date)
);
`;

async function initializeDatabase() {
  let connection;
  
  try {
    console.log('🔄 Connecting to MySQL server...');
    
    // Connect without database first
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      port: process.env.DB_PORT || 3306,
      multipleStatements: true
    });

    console.log('✓ Connected to MySQL server');
    console.log('🔄 Creating database and tables...');

    // Execute schema
    await connection.query(SQL_SCHEMA);
    console.log('✓ Database schema created successfully');

    // Create default superadmin
    console.log('🔄 Creating default superadmin...');
    
    const hashedPassword = await bcrypt.hash(
      process.env.SUPER_ADMIN_PASSWORD || 'Admin@123',
      10
    );

    const insertSuperAdmin = `
      INSERT INTO ${process.env.DB_NAME || 'faceml_db'}.users 
      (email, password, full_name, role) 
      VALUES (?, ?, 'Super Admin', 'superadmin')
      ON DUPLICATE KEY UPDATE password = VALUES(password), updated_at = CURRENT_TIMESTAMP;
    `;

    await connection.query(insertSuperAdmin, [
      process.env.SUPER_ADMIN_EMAIL || 'admin@faceml.com',
      hashedPassword
    ]);

    console.log('✓ Default superadmin created/updated');
    console.log('\n✅ Database initialization completed successfully!\n');
    console.log('Default Login Credentials:');
    console.log('Email:', process.env.SUPER_ADMIN_EMAIL || 'admin@faceml.com');
    console.log('Password:', process.env.SUPER_ADMIN_PASSWORD || 'Admin@123');
    console.log('\n⚠️  Please change the default password after first login!\n');

  } catch (error) {
    console.error('❌ Database initialization failed:', error.message);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

initializeDatabase();

