/**
 * Safe migration script to convert time fields to VARCHAR
 * This script handles existing records carefully and provides rollback capability
 * 
 * Usage: node migrate_time_to_string_safe.js
 */

const db = require('../config/database');

async function migrate() {
  try {
    console.log('🔄 Starting time field migration to VARCHAR...\n');

    // Step 1: Check current schema
    console.log('📋 Checking current schema...');
    const [attendanceColumns] = await db.query(`
      SELECT COLUMN_NAME, DATA_TYPE, COLUMN_TYPE 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'attendance' 
      AND COLUMN_NAME IN ('clock_in_time', 'clock_out_time')
    `);

    console.log('Current attendance time columns:');
    attendanceColumns.forEach(col => {
      console.log(`  - ${col.COLUMN_NAME}: ${col.DATA_TYPE} (${col.COLUMN_TYPE})`);
    });

    // Step 2: Backup existing data count
    const [attendanceCount] = await db.query('SELECT COUNT(*) as count FROM attendance WHERE clock_in_time IS NOT NULL');
    const [workSessionsCount] = await db.query('SELECT COUNT(*) as count FROM work_sessions WHERE start_time IS NOT NULL');
    
    console.log(`\n📊 Records to migrate:`);
    console.log(`  - Attendance records with clock_in_time: ${attendanceCount[0].count}`);
    console.log(`  - Work sessions with start_time: ${workSessionsCount[0].count}`);

    // Step 3: Migrate attendance table
    console.log('\n🔄 Migrating attendance table...');
    
    // Add temporary columns
    await db.query(`
      ALTER TABLE attendance 
      ADD COLUMN clock_in_time_str VARCHAR(19) NULL,
      ADD COLUMN clock_out_time_str VARCHAR(19) NULL
    `);
    console.log('  ✅ Added temporary columns');

    // Migrate data
    await db.query(`
      UPDATE attendance 
      SET clock_in_time_str = DATE_FORMAT(clock_in_time, '%Y-%m-%d %H:%i:%s')
      WHERE clock_in_time IS NOT NULL
    `);
    console.log(`  ✅ Migrated ${attendanceCount[0].count} clock_in_time records`);

    await db.query(`
      UPDATE attendance 
      SET clock_out_time_str = DATE_FORMAT(clock_out_time, '%Y-%m-%d %H:%i:%s')
      WHERE clock_out_time IS NOT NULL
    `);
    const [clockOutCount] = await db.query('SELECT COUNT(*) as count FROM attendance WHERE clock_out_time IS NOT NULL');
    console.log(`  ✅ Migrated ${clockOutCount[0].count} clock_out_time records`);

    // Drop old columns and rename new ones
    await db.query(`ALTER TABLE attendance DROP COLUMN clock_in_time`);
    await db.query(`ALTER TABLE attendance DROP COLUMN clock_out_time`);
    await db.query(`ALTER TABLE attendance CHANGE COLUMN clock_in_time_str clock_in_time VARCHAR(19) NULL`);
    await db.query(`ALTER TABLE attendance CHANGE COLUMN clock_out_time_str clock_out_time VARCHAR(19) NULL`);
    console.log('  ✅ Renamed columns');

    // Step 4: Migrate work_sessions table
    console.log('\n🔄 Migrating work_sessions table...');
    
    // Add temporary columns
    await db.query(`
      ALTER TABLE work_sessions 
      ADD COLUMN start_time_str VARCHAR(19) NULL,
      ADD COLUMN end_time_str VARCHAR(19) NULL
    `);
    console.log('  ✅ Added temporary columns');

    // Migrate data
    await db.query(`
      UPDATE work_sessions 
      SET start_time_str = DATE_FORMAT(start_time, '%Y-%m-%d %H:%i:%s')
      WHERE start_time IS NOT NULL
    `);
    console.log(`  ✅ Migrated ${workSessionsCount[0].count} start_time records`);

    await db.query(`
      UPDATE work_sessions 
      SET end_time_str = DATE_FORMAT(end_time, '%Y-%m-%d %H:%i:%s')
      WHERE end_time IS NOT NULL
    `);
    const [endTimeCount] = await db.query('SELECT COUNT(*) as count FROM work_sessions WHERE end_time IS NOT NULL');
    console.log(`  ✅ Migrated ${endTimeCount[0].count} end_time records`);

    // Drop old columns and rename new ones
    await db.query(`ALTER TABLE work_sessions DROP COLUMN start_time`);
    await db.query(`ALTER TABLE work_sessions DROP COLUMN end_time`);
    await db.query(`ALTER TABLE work_sessions CHANGE COLUMN start_time_str start_time VARCHAR(19) NULL`);
    await db.query(`ALTER TABLE work_sessions CHANGE COLUMN end_time_str end_time VARCHAR(19) NULL`);
    console.log('  ✅ Renamed columns');

    // Step 5: Verify migration
    console.log('\n✅ Verifying migration...');
    const [verifyAttendance] = await db.query(`
      SELECT COUNT(*) as count 
      FROM attendance 
      WHERE clock_in_time IS NOT NULL 
      AND clock_in_time REGEXP '^[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}:[0-9]{2}$'
    `);
    console.log(`  ✅ Verified ${verifyAttendance[0].count} attendance records have correct format`);

    console.log('\n✅ Migration completed successfully!');
    console.log('\n📝 Note: All time fields are now stored as VARCHAR(19) in format: YYYY-MM-DD HH:MM:SS');
    console.log('   This preserves local time without timezone conversion.\n');

  } catch (error) {
    console.error('❌ Migration error:', error);
    console.error('\n⚠️  If migration failed partway through, you may need to manually fix the schema.');
    throw error;
  }
}

// Run migration
if (require.main === module) {
  migrate()
    .then(() => {
      console.log('Migration script completed');
      process.exit(0);
    })
    .catch((error) => {
      console.error('Migration failed:', error);
      process.exit(1);
    });
}

module.exports = { migrate };

