const mysql = require('mysql2/promise');
const fs = require('fs').promises;
const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '..', '.env') });

/**
 * Restores staff data from backup
 */
async function restoreFromBackup() {
  let connection;
  
  try {
    // Get the most recent backup file
    const backupDir = path.join(__dirname, '..', 'backups');
    const files = await fs.readdir(backupDir);
    const backupFiles = files.filter(f => f.startsWith('staff_backup_') && f.endsWith('.json'));
    
    if (backupFiles.length === 0) {
      console.log('❌ No backup files found');
      return;
    }
    
    // Sort by timestamp (filename) and get the most recent
    backupFiles.sort().reverse();
    const latestBackup = path.join(backupDir, backupFiles[0]);
    
    console.log(`🔄 Restoring from: ${latestBackup}`);
    
    // Read backup file
    const backupData = await fs.readFile(latestBackup, 'utf8');
    const staffRecords = JSON.parse(backupData);
    
    console.log(`✓ Found ${staffRecords.length} records in backup`);
    
    // Connect to database
    console.log('🔄 Connecting to database...');
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
    });
    
    console.log('✓ Connected to database');
    console.log('🔄 Restoring staff records...');
    
    let successCount = 0;
    let errorCount = 0;
    
    for (const staff of staffRecords) {
      try {
        await connection.query(
          `UPDATE staff 
           SET full_name = ?, 
               email = ?, 
               phone = ?, 
               face_embeddings = ?
           WHERE id = ?`,
          [
            staff.full_name,
            staff.email,
            staff.phone,
            staff.face_embeddings ? JSON.stringify(staff.face_embeddings) : null,
            staff.id
          ]
        );
        
        successCount++;
        console.log(`  ✓ Restored staff ID ${staff.id}`);
      } catch (error) {
        errorCount++;
        console.error(`  ❌ Failed to restore staff ID ${staff.id}:`, error.message);
      }
    }
    
    console.log('\n✅ Restore completed!');
    console.log(`  Successfully restored: ${successCount} record(s)`);
    console.log(`  Errors: ${errorCount} record(s)\n`);
    
  } catch (error) {
    console.error('❌ Restore failed:', error.message);
    console.error('Full error:', error);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('Database connection closed.');
    }
  }
}

// Run the restore
restoreFromBackup();
