const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

async function runMigration() {
  let connection;
  
  try {
    // Create database connection
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
      multipleStatements: true
    });

    console.log('✓ Connected to database');

    // Read SQL migration file
    const sqlFile = path.join(__dirname, 'create_feature_flags_table.sql');
    const sql = fs.readFileSync(sqlFile, 'utf8');

    console.log('✓ Reading migration file...');

    // Execute migration
    await connection.query(sql);
    
    console.log('✓ Migration executed successfully!');
    console.log('✓ Feature flags table created/updated');
    console.log('✓ Default flags (roster, availability) inserted');

    // Verify the table was created
    const [tables] = await connection.query(
      "SHOW TABLES LIKE 'feature_flags'"
    );
    
    if (tables.length > 0) {
      console.log('✓ Feature flags table exists');
      
      // Check flags
      const [flags] = await connection.query(
        'SELECT feature_name, is_enabled FROM feature_flags'
      );
      
      console.log('\n✓ Current feature flags:');
      flags.forEach(flag => {
        console.log(`  - ${flag.feature_name}: ${flag.is_enabled ? 'ENABLED' : 'DISABLED'}`);
      });
    }

  } catch (error) {
    console.error('✗ Migration failed:', error.message);
    process.exit(1);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\n✓ Database connection closed');
    }
  }
}

runMigration();

