const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');
require('dotenv').config();

async function runMigration() {
  let connection;
  
  try {
    // Create database connection
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306,
      multipleStatements: true
    });

    console.log('✓ Connected to database');

    // Read SQL migration file
    const sqlFile = path.join(__dirname, 'alter_feature_flags_shop_wise.sql');
    const sql = fs.readFileSync(sqlFile, 'utf8');

    console.log('✓ Reading migration file...');

    // Execute migration
    await connection.query(sql);
    
    console.log('✓ Migration executed successfully!');
    console.log('✓ Feature flags table altered to support shop-specific flags');

    // Verify the table columns
    const [columns] = await connection.query(
      "SHOW COLUMNS FROM feature_flags"
    );
    
    console.log('\n✓ Table columns:');
    columns.forEach(col => {
      console.log(`  - ${col.Field}: ${col.Type}`);
    });

  } catch (error) {
    console.error('✗ Migration failed:', error.message);
    // Don't exit with error if it's a duplicate/already exists error
    const skipErrors = [
      "Duplicate column name",
      "Duplicate key name",
      "Duplicate foreign key",
      "already exists"
    ];
    
    const isSkipError = skipErrors.some(err => 
      error.message.includes(err) || error.code === 'ER_DUP_FIELDNAME' || error.code === 'ER_DUP_KEYNAME'
    );
    
    if (!isSkipError) {
        console.error('✗ Fatal error - migration failed');
        process.exit(1);
    } else {
        console.log('  (Migration likely already ran - some elements already exist)');
    }
  } finally {
    if (connection) {
      await connection.end();
      console.log('\n✓ Database connection closed');
    }
  }
}

runMigration();
