const mysql = require('mysql2/promise');
require('dotenv').config();

async function testShopAdmin() {
  let connection;
  
  try {
    console.log('🔄 Connecting to database...');
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'faceml_db',
      port: process.env.DB_PORT || 3306
    });

    console.log('✓ Connected!\n');

    // Check all users
    console.log('📋 All Users:');
    const [users] = await connection.query(
      'SELECT id, email, full_name, role, is_active FROM users ORDER BY id'
    );
    console.table(users);

    // Check all shops
    console.log('\n🏪 All Shops:');
    const [shops] = await connection.query(
      'SELECT id, name, admin_id, address, is_active FROM shops ORDER BY id'
    );
    console.table(shops);

    // Check all branches
    console.log('\n🏢 All Branches:');
    const [branches] = await connection.query(
      'SELECT id, shop_id, name, address, is_active FROM branches ORDER BY id'
    );
    console.table(branches);

    // Check shop admins with their shops and branches
    console.log('\n👥 Shop Admins with Shops and Branches:');
    const [adminShops] = await connection.query(`
      SELECT 
        u.id as user_id,
        u.email,
        u.full_name,
        s.id as shop_id,
        s.name as shop_name,
        b.id as branch_id,
        b.name as branch_name,
        b.is_active as branch_active
      FROM users u
      LEFT JOIN shops s ON u.id = s.admin_id
      LEFT JOIN branches b ON s.id = b.shop_id
      WHERE u.role = 'shop_admin'
      ORDER BY u.id, s.id, b.id
    `);
    console.table(adminShops);

    // Summary
    console.log('\n📊 Summary:');
    console.log(`Total Users: ${users.length}`);
    console.log(`Total Shops: ${shops.length}`);
    console.log(`Total Branches: ${branches.length}`);
    console.log(`Shop Admins: ${users.filter(u => u.role === 'shop_admin').length}`);
    console.log(`Superadmins: ${users.filter(u => u.role === 'superadmin').length}`);

  } catch (error) {
    console.error('❌ Error:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\n✓ Connection closed');
    }
  }
}

testShopAdmin();

