const express = require('express');
const cors = require('cors');
require('dotenv').config();

const authRoutes = require('./routes/authRoutes');
const shopAdminRoutes = require('./routes/shopAdminRoutes');
const branchRoutes = require('./routes/branchRoutes');
const staffRoutes = require('./routes/staffRoutes');
const attendanceRoutes = require('./routes/attendanceRoutes');
const availabilityRoutes = require('./routes/availabilityRoutes');
const rosterRoutes = require('./routes/rosterRoutes');
const featureFlagsRoutes = require('./routes/featureFlagsRoutes');
const subscriptionRoutes = require('./routes/subscriptionRoutes');
const userRoutes = require('./routes/userRoutes');
const { autoClockoutExpiredSessions } = require('./services/autoClockoutService');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());

// Stripe webhook route needs raw body for signature verification
// This must be before express.json() middleware
app.use('/api/v1/stripe-webhook', express.raw({ type: 'application/json' }));

app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Request logging middleware
app.use((req, res, next) => {
  console.log(`${new Date().toISOString()} - ${req.method} ${req.path}`);
  next();
});

// Health check
app.get('/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    environment: process.env.NODE_ENV || 'development'
  });
});

// API Routes
app.use('/api/auth', authRoutes);
app.use('/api/shop-admins', shopAdminRoutes);
app.use('/api/branches', branchRoutes);
app.use('/api/staff', staffRoutes);
app.use('/api/attendance', attendanceRoutes);
app.use('/api/availability', availabilityRoutes);
app.use('/api/roster', rosterRoutes);
app.use('/api/feature-flags', featureFlagsRoutes);
app.use('/api/v1', subscriptionRoutes);
app.use('/api/v1', userRoutes);

// 404 handler
app.use((req, res) => {
  res.status(404).json({ error: 'Route not found' });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Error:', err);
  res.status(err.status || 500).json({ 
    error: err.message || 'Internal server error' 
  });
});

// Start server
app.listen(PORT, () => {
  console.log('='.repeat(50));
  console.log(`🚀 FaceML Backend API Server`);
  console.log('='.repeat(50));
  console.log(`📍 Server running on: http://localhost:${PORT}`);
  console.log(`🌍 Environment: ${process.env.NODE_ENV || 'development'}`);
  console.log(`📅 Started at: ${new Date().toISOString()}`);
  console.log('='.repeat(50));
  console.log('\nAvailable endpoints:');
  console.log('  GET  /health');
  console.log('  POST /api/auth/login');
  console.log('  GET  /api/auth/profile');
  console.log('  POST /api/shop-admins');
  console.log('  POST /api/branches');
  console.log('  POST /api/staff');
  console.log('  POST /api/attendance/clock-in');
  console.log('='.repeat(50));
  
  // Start auto-clockout scheduler
  // Run every hour to check for expired sessions
  console.log('\n⏰ Starting auto-clockout scheduler (runs every hour)...');
  autoClockoutExpiredSessions(); // Run immediately on startup
  setInterval(() => {
    autoClockoutExpiredSessions();
  }, 60 * 60 * 1000); // Run every hour (60 minutes * 60 seconds * 1000 ms)
});

module.exports = app;

