const db = require('../config/database');

/**
 * Automatically clock out staff who have been clocked in for more than 12 hours
 * This runs periodically to check for forgotten clockouts
 */
async function autoClockoutExpiredSessions() {
  try {
    console.log('\n========================================');
    console.log('🔄 AUTO-CLOCKOUT JOB STARTED');
    console.log(`🕐 Server Time: ${new Date().toISOString()}`);
    console.log('========================================\n');

    // Find all attendance records where:
    // 1. clock_in_time is set
    // 2. clock_out_time is NULL (still clocked in)
    // 3. clock_in_time is more than 12 hours ago
    // 4. is_automatic_clockout is NULL or FALSE (not already auto-clocked out)
    
    const twelveHoursAgo = new Date();
    twelveHoursAgo.setHours(twelveHoursAgo.getHours() - 12);
    
    // Format as string for MySQL (YYYY-MM-DD HH:MM:SS)
    const twelveHoursAgoStr = twelveHoursAgo.toISOString().slice(0, 19).replace('T', ' ');

    const [expiredSessions] = await db.query(
      `SELECT 
        id, 
        staff_id, 
        branch_id,
        clock_in_time,
        attendance_date
      FROM attendance 
      WHERE clock_in_time IS NOT NULL 
        AND clock_out_time IS NULL 
        AND clock_in_time <= ?
        AND (is_automatic_clockout IS NULL OR is_automatic_clockout = FALSE)
      ORDER BY clock_in_time ASC`,
      [twelveHoursAgoStr]
    );

    console.log(`📋 Found ${expiredSessions.length} expired sessions (over 12 hours)`);

    if (expiredSessions.length === 0) {
      console.log('✅ No expired sessions to auto-clockout\n');
      return;
    }

    let successCount = 0;
    let errorCount = 0;

    for (const session of expiredSessions) {
      try {
        // Calculate automatic clockout time (12 hours after clock-in)
        // IMPORTANT: clock_in_time is stored as local time string "YYYY-MM-DD HH:MM:SS"
        // We need to parse it as local time and add 12 hours, then format back as local time string
        const clockInTimeStr = String(session.clock_in_time);
        
        // Parse the local time string manually (treat as local time, not UTC)
        // Format: "YYYY-MM-DD HH:MM:SS"
        const [datePart, timePart] = clockInTimeStr.split(' ');
        const [year, month, day] = datePart.split('-').map(Number);
        const [hours, minutes, seconds] = timePart.split(':').map(Number);
        
        // Create Date object in local timezone
        const clockInTime = new Date(year, month - 1, day, hours, minutes, seconds || 0);
        
        // Add 12 hours
        const autoClockOutTime = new Date(clockInTime);
        autoClockOutTime.setHours(autoClockOutTime.getHours() + 12);
        
        // Format back as local time string (YYYY-MM-DD HH:MM:SS) - NOT UTC
        const yearOut = autoClockOutTime.getFullYear();
        const monthOut = String(autoClockOutTime.getMonth() + 1).padStart(2, '0');
        const dayOut = String(autoClockOutTime.getDate()).padStart(2, '0');
        const hoursOut = String(autoClockOutTime.getHours()).padStart(2, '0');
        const minutesOut = String(autoClockOutTime.getMinutes()).padStart(2, '0');
        const secondsOut = String(autoClockOutTime.getSeconds()).padStart(2, '0');
        
        const autoClockOutTimeStr = `${yearOut}-${monthOut}-${dayOut} ${hoursOut}:${minutesOut}:${secondsOut}`;

        // Calculate total hours (will be 0 since it's automatic, but we'll set it to 0 explicitly)
        const totalHours = 0;

        // Update attendance record with automatic clockout
        await db.query(
          `UPDATE attendance 
           SET clock_out_time = ?,
               is_automatic_clockout = TRUE,
               total_hours = ?
           WHERE id = ?`,
          [autoClockOutTimeStr, totalHours, session.id]
        );

        console.log(`✅ Auto-clocked out session ID: ${session.id}, Staff ID: ${session.staff_id}`);
        console.log(`   Clock In: ${session.clock_in_time}`);
        console.log(`   Auto Clock Out: ${autoClockOutTimeStr}`);
        console.log(`   Total Hours: 0 (not counted as work time)`);
        
        successCount++;
      } catch (error) {
        console.error(`❌ Error auto-clocking out session ID ${session.id}:`, error.message);
        errorCount++;
      }
    }

    console.log('\n========================================');
    console.log('📊 AUTO-CLOCKOUT JOB SUMMARY');
    console.log(`✅ Successfully processed: ${successCount}`);
    console.log(`❌ Errors: ${errorCount}`);
    console.log('========================================\n');

  } catch (error) {
    console.error('❌ Auto-clockout job error:', error);
  }
}

module.exports = {
  autoClockoutExpiredSessions
};

